rm(list=ls())
library("QREM")
# https://github.com/haimbar/QREM

# Mixed Effect Modeling and Variable Selection for Quantile Regression
# Bar, Booth, and Wells (Statistical Modelling 2021)

# Code to create Figure 2 (and 9A and 9B)

library(quantreg)

# change B=0 to a positive number (e.g. 30) below if you want to include
# bootstrap estimates (it takes *considerably* longer)

# For figure 2:
sims <- list(seed=11001, n=1000,  err=function(x) {rnorm(length(x), 0, 0.1)},
             errvar=0, mod=y~X1, coefs=c(-3,1),
             qns=c(0.02,seq(0.05,0.95,by=0.05),0.98),
             B=0, reps=100, xrng=rbind(c(-5,5)))

# # For figure 9A
# sims <- list(seed=11001, n=1000,  err=function(x) {rnorm(length(x), 0, 0.1+0.5*x)},
#              errvar=1, mod=y~X1, coefs=c(5,1),
#              qns=c(0.02,seq(0.05,0.95,by=0.05),0.98),
#              B=0, reps=100, xrng=rbind(c(0,1)))
# # For figure 9B
# sims <- list(seed=11001, n=1000,  err=function(x) {rlnorm(length(x), 0, 0.75)},
#              errvar=0, mod=y~X1+X2+X3+X4+X5, coefs=c(1,-3,2,2,-1,-2),
#              qns=c(0.02,seq(0.05,0.95,by=0.05),0.98),
#              B=0, reps=100, xrng=rbind(c(-1,1),c(-1,1),c(-1,1),c(-1,1),c(-1,1)))

n <- sims$n
qns <- sims$qns
reps <- sims$reps
coefs <- sims$coefs
xrng <- matrix(sims$xrng,  nrow=(length(coefs)-1), ncol=2)
cnt <- 1
res <- matrix(0,nrow=length(qns)*reps, ncol=(2+7*length(coefs)))
cnames <- c("qn","rep",
            paste("bQREM",0:(length(coefs)-1),  sep=""),
            paste("brq",0:(length(coefs)-1),  sep=""),
            paste("blm",0:(length(coefs)-1),  sep=""),
            paste("sdQREM",0:(length(coefs)-1),  sep=""),
            paste("sdrq",0:(length(coefs)-1),  sep=""),
            paste("sdlm",0:(length(coefs)-1),  sep=""),
            paste("sdBoot",0:(length(coefs)-1),  sep=""))
pb <- txtProgressBar(style=3)
for (qn in qns) {
  for (i in 1:reps) {
    set.seed(23*100000 + sims$seed + i)
    X <- matrix(0,nrow=n, ncol=length(coefs))
    X[,1] <- rep(1,n)
    for (jj in 1:(length(coefs)-1)) {
      X[,jj+1] <- runif(n, min=xrng[jj,1], max=xrng[jj,2])
    }
    colnames(X) <- c("const", paste("X",1:(length(coefs)-1),sep=""))
    if (sims$errvar == 0) {
      errs <- sims$err(rep(0,n))
    } else {
      errs <- sims$err(X[,sims$errvar+1])
    }
    y <- X%*%coefs + errs
    dframe <- data.frame(y,X)
    
    qremFit <- QREM("lm",linmod=sims$mod, dframe=dframe, qn=qn)
    varKED <- diag(bcov(qremFit, sims$mod, dframe, qn))
    varBS <- rep(0,length(coefs))
    if (sims$B > 0) {
      estBS <- boot.QREM(func="lm", linmod=sims$mod, df = dframe, qn=qn,
                         n=n, sampleFrom=NULL,B=sims$B)
      varBS <- apply(estBS, 2, var)
    }
    rqFit <- rq(sims$mod, tau = qn, data = dframe)
    varRQ <- ((summary(rqFit)$coef[,3]-summary(rqFit)$coef[,2])/(2*qnorm(0.95)))^2
    lmf <- lm(sims$mod, data = dframe)
    varLM <- (summary(lmf)$coefficients[,2])^2
    res[cnt,] <- c(qn, i, qremFit$coef$beta, rqFit$coefficients, lmf$coefficients,
                   sqrt(varKED), sqrt(varRQ), sqrt(varLM), sqrt(varBS))
    cnt <- cnt+1
    setTxtProgressBar(pb, qn)
  }
}
res <- data.frame(res)
colnames(res) <- cnames
close(pb)

if (sims$B == 0) {
  plot(res$qn-0.007, res$sdQREM1,axes=FALSE, xlab="quantile",ylab="s.d.",
       pch=15, cex=0.5, col="navyblue", ylim=c(0,max(res$sdQREM1,res$sdrq1)))
  points(res$qn+0.007, res$sdrq1,col="chocolate1",cex=0.5, pch=16)
  axis(1); axis(2)
  abline(v=qns,lty=3,col="grey")
  legend("top",legend = c("kernel-based (QREM)","rank-based (quantreg)"), 
         col=c("navyblue","chocolate1"),y.intersp=0.2,
         pch=c(15,16), bty = "o", bg="transparent", box.col="white")
} else { # show bootstrap sd estimates:
  plot(res$qn-0.013, res$sdQREM1,axes=FALSE, xlab="quantile",ylab="s.d.",
       pch=15, cex=0.5, col="navyblue", ylim=c(0,max(res$sdQREM1,res$sdrq1)))
  abline(v=qns,lty=3,col="grey")
  points(res$qn, res$sdrq1,col="darkred",cex=0.5, pch=17)
  points(res$qn+0.013, res$sdBoot1,col="chocolate1",cex=0.5, pch=16)
  abline(h=0)
  text(seq(0.1,0.9,by=0.1),rep(0.0003,9),seq(0.1,0.9,by=0.1)); 
  axis(2)
  legend("top",legend = c("kernel-based (QREM)","rank-based (quantreg)","bootstrap"), 
         col=c("navyblue","darkred","chocolate1"),y.intersp=0.2,
         pch=c(15,17, 16), bty = "o", bg="transparent", box.col="white")
}
