//following functions are from the web: https://mc-stan.org/users/documentation/case-studies/mbjoseph-CARStan.html
functions {
  /**
  * Return the log probability of a proper conditional autoregressive (CAR) prior 
  * with a sparse representation for the adjacency matrix
  *
  * @param phi Vector containing the parameters with a CAR prior
  * @param tau Precision parameter for the CAR prior (real)
  * @param alpha Dependence (usually spatial) parameter for the CAR prior (real)
  * @param W_sparse Sparse representation of adjacency matrix (int array)
  * @param n Length of phi (int)
  * @param W_n Number of adjacent pairs (int)
  * @param D_sparse Number of neighbors for each location (vector)
  * @param lambda Eigenvalues of D^{-1/2}*W*D^{-1/2} (vector)
  *
  * @return Log probability density of CAR prior up to additive constant
  */
  real sparse_car_lpdf(vector phi, real tau, real alpha, 
    int[,] W_sparse, vector D_sparse, vector lambda, int n, int W_n) {
      row_vector[n] phit_D; // phi' * D
      row_vector[n] phit_W; // phi' * W
      vector[n] ldet_terms;
    
      phit_D = (phi .* D_sparse)';
      phit_W = rep_row_vector(0, n);
      for (i in 1:W_n) {
        phit_W[W_sparse[i, 1]] = phit_W[W_sparse[i, 1]] + phi[W_sparse[i, 2]];
        phit_W[W_sparse[i, 2]] = phit_W[W_sparse[i, 2]] + phi[W_sparse[i, 1]];
      }
    
      for (i in 1:n) ldet_terms[i] = log1m(alpha * lambda[i]);
      return 0.5 * (n * log(tau)
                    + sum(ldet_terms)
                    - tau * (phit_D * phi - alpha * (phit_W * phi)));
  }
}


data {
  int<lower=1> n;
  int<lower=1> N;
  int y[N];
  vector[N] pop_tn;
  int<lower=1> K; //number of predictors for poisson part
  matrix[N, K] x_all; 
  matrix<lower = 0, upper = 1>[n, n] W; // adjacency matrix
  int W_n;                // number of adjacent region pairs
  real gamma1;
  real gamma2;
}


transformed data {
  int W_sparse[W_n, 2];   // adjacency pairs
  vector[n] D_sparse;     // diagonal of D (number of neigbors for each site)
  vector[n] lambda;       // eigenvalues of invsqrtD * W * invsqrtD

  { // generate sparse representation for W
  int counter;
  counter = 1;
  // loop over upper triangular part of W to identify neighbor pairs
    for (i in 1:(n - 1)) {
      for (j in (i + 1):n) {
        if (W[i, j] == 1) {
          W_sparse[counter, 1] = i;
          W_sparse[counter, 2] = j;
          counter = counter + 1;
        }
      }
    }
  }
  for (i in 1:n) D_sparse[i] = sum(W[i]);
  {
    vector[n] invsqrtD;  
    for (i in 1:n) {
      invsqrtD[i] = 1 / sqrt(D_sparse[i]);
    }
    lambda = eigenvalues_sym(quad_form(W, diag_matrix(invsqrtD)));
  }  
}


parameters {
  vector[K] beta_m;     // coefficients in Poisson part
  vector[n] phi;
  real<lower = 0> sd_tau;
  real<lower = 0, upper = 1> alpha;  
}


transformed parameters {
  real <lower = 0> tau;
  tau = inv(square(sd_tau));
 }


model{
  vector[N] m_log;   
  vector[N] m;  
  m_log = x_all * beta_m ;   
  m=exp(m_log + log(pop_tn) + to_vector(rep_matrix(phi, 14)));
    for (i in 1:N) { 
      y[i] ~ poisson(m[i]);
    } 
  //alpha ~ uniform(0,1);
  alpha ~ beta(gamma1,gamma2);
  sd_tau ~ uniform(0,10);
  phi ~ sparse_car(tau, alpha, W_sparse, D_sparse, lambda, n, W_n);
  beta_m ~ multi_normal(rep_vector(0, K), diag_matrix(rep_vector(100, K)));
}


generated quantities {
  vector[N] sim_m; 
  vector[N] sim_m_log; 
  int y_rep[N]; 
  vector[N] log_lik;
  sim_m_log = x_all * beta_m;   
  sim_m = exp(sim_m_log + log(pop_tn) + to_vector(rep_matrix(phi, 14)));
  for (i in 1:N) { 
          y_rep[i] = poisson_rng(sim_m[i]);
          log_lik[i] = poisson_lpmf(y[i] | sim_m[i]);
      } 
}
