# EMdatastandard
# CalSigma - inner function in EMdatastandard

EMdatastandard=function(dataset,K)
{
  #
  # Standard approach: fitting function
  #
  # Input:
  # dataset: data.frame
  # K: number of locations
  #
  # Output:
  # beta0: intercepts (used for computing MCOV and MCOR) 
  # beta: parameter vector estimate
  # u: estimated intercepts (locations)
  # p: estimated prior probabilities
  # likeli: likelihood value upon convergence
  # MCOV: covariance matrix for the random effects
  # MCOR: correlation matrix for the random effects
  #
  #data
  n=dim(dataset)[1]
  x1=x2=cbind(dataset$X1,dataset$X2,dataset$X3,dataset$X4,dataset$X8,dataset$X15,dataset$X16)
  x3=cbind(dataset$X7,dataset$X9,dataset$X10,dataset$X11,dataset$X12,dataset$X14)
  y1=dataset$Y5
  y2=dataset$Y6
  y3=dataset$X15

  #set-up matrices and vectors 
  d=matrix(rep(1,K))
  y1m=rep(y1,K)
  y2m=rep(y2,K)
  y3m=rep(y3,K)
  x1m=kronecker(d,x1)
  x2m=kronecker(d,x2)
  x3m=kronecker(d,x3)
  onesk=matrix(1,1,K)

  #weights
  wi=matrix(runif(n*K),n,K)
  wi=wi/(apply(wi,c(1),sum)%*%onesk)
  pk=apply(wi,c(2),mean)
  z=matrix(kronecker(diag(rep(1,K)),rep(1,n)),n*K,K)

  #conditional model 1
  mod1=glm(y1m~x1m+z-1,family=poisson,weights=c(wi))
  beta1=mod1$coef
  mi1=mod1$fitted
  phi1=dpois(y1m,mi1)
  fi1=array(ifelse(phi1>0,phi1,1e-005),c(n,K))

  #conditional model 2
  mod2=glm(y2m~x2m+z-1,family=poisson,weights=c(wi))
  beta2=mod2$coef
  mi2=mod2$fitted
  phi2=dpois(y2m,mi2)
  fi2=array(ifelse(phi2>0,phi2,1e-005),c(n,K))

  #conditional model 3
  mod3=glm(cbind(y3m,1-y3m)~x3m+z-1,family=binomial,weights=c(wi))
  beta3=mod3$coef
  mi3=mod3$fitted
  phi3=dbinom(y3m,1,mi3)
  fi3=array(ifelse(phi3>0,phi3,1e-005),c(n,K))

  # joint density and log-likelihood
  fi=fi1*fi2*fi3
  li=fi%*%diag(pk)
  wi=li/(matrix(fi%*% pk,n,K))
  pk=apply(wi,c(2),mean)
  li=log(apply(li,c(1),sum))
  likeli=sum(li)

  #iterative cycle
  diff=1
  niter=1
  while(diff>1e-007)
  {
    likeold<-likeli
  
    #conditional model 1
    mod1=glm(y1m~x1m+z-1,family=poisson,weights=c(wi))
    beta1=mod1$coef
    mi1=mod1$fitted
    phi1=dpois(y1m,mi1)
    fi1=array(ifelse(phi1>0,phi1,1e-005),c(n,K))

    #conditional model 2
    mod2=glm(y2m~x2m+z-1,family=poisson,weights=c(wi))
    beta2=mod2$coef
    mi2=mod2$fitted
    phi2=dpois(y2m,mi2)
    fi2=array(ifelse(phi2>0,phi2,1e-005),c(n,K))

    #conditional model 3
    mod3=glm(cbind(y3m,1-y3m)~x3m+z-1,family=binomial,weights=c(wi))
    beta3=mod3$coef
    mi3=mod3$fitted
    phi3=dbinom(y3m,1,mi3)
    fi3=array(ifelse(phi3>0,phi3,1e-005),c(n,K))

    # joint density and log-likelihood
    fi=fi1*fi2*fi3
    pk=apply(wi,c(2),mean)
    lik=fi%*%diag(pk)
    li=log(apply(lik,c(1),sum))
    likeli=sum(li)
    wi=lik/(matrix(fi%*%pk,n,K))
    diff=(likeli-likeold)/abs(likeold)
    niter=niter+1
  }

  #parameter estimates
  SS=CalSigma(beta1,beta2,beta3,K,pk)
  betag=c(beta1[1],beta2[1],beta3[1]) 
  ug=c(beta1[2:length(beta1)],beta2[2:length(beta2)],beta3[2:length(beta3)])
  if (is.null(ncol(x1))==F)
  {
    betag=c(beta1[1:(ncol(x1))],beta2[1:(ncol(x2))],beta3[1:(ncol(x3))]) 
    ug=c(beta1[(ncol(x1)+1):length(beta1)],beta2[(ncol(x2)+1):length(beta2)],beta3[(ncol(x3)+1):length(beta3)])
  }

  #results
  resul=list()
  resul$beta0=SS$beta0
  resul$beta=betag
  resul$u=ug
  resul$p=pk
  resul$likeli=likeli
  resul$MCOV=SS$cov
  resul$MCOR=SS$cor
  return(resul)
}

CalSigma=function(par1,par2,par3,K,gpi)
{
  # inner function in EMdatastandard
  K=length(gpi)
  S=3
  beta1=par1[1:(length(par1)-K)]
  beta2=par2[1:(length(par2)-K)]
  beta3=par3[1:(length(par3)-K)]
  Beta=c(beta1,beta2,beta3)
  u1=par1[(length(beta1)+1):length(par1)]
  u2=par2[(length(beta2)+1):length(par2)]
  u3=par3[(length(beta3)+1):length(par3)]
  U=cbind(u1,u2,u3)
  Beta0=t(gpi)%*%U
  Cova=cov.wt(U, wt=c(gpi), cor=T)
  Mcov=Cova$cov
  Mcor=Cova$cor
  res=list()
  res$beta=Beta
  res$beta0=Beta0
  res$cov=Mcov
  res$cor=Mcor
  return(res)
}