# EMdataparafac
# EMdataparafacseq
# CPprob - inner function in EMdataparafac and EMdataparafacseq
# CalSigma2 - inner function in EMdataparafac and EMdataparafacseq
# updateCOL - inner function in CPprob

EMdataparafac=function(dataset,Kappa,qqq) 
{
  #
  # Parafac approach: fitting function (iterative procedure)
  #
  # Input:
  # dataset: data.frame
  # Kappa: vector with the number of locations in each profile (K1, K2, K3)
  # qqq: number of components in Parafac
  #
  # Output:
  # beta0: intercepts (used for computing MCOV and MCOR) 
  # beta: parameter vector estimate
  # K1: number of locations in profile 1 
  # K2: number of locations in profile 2
  # K3: number of locations in profile 3
  # u: estimated intercepts (locations)
  # pi: estimated joint probabilities (tensor)
  # pirid: Parafac approximation of the joint probabilities (tensor)
  # MCOV: covariance matrix for the random effects
  # post: posterior probabilities
  # likeli: likelihood value upon convergence
  # Aparafac: estimated matrix P1
  # Bparafac: estimated matrix P2
  # Cparafac: estimated matrix P3
  # Hparafac: estimated matrix T
  # mod1: conditional model 1
  # mod2: conditional model 2
  # mod3: conditional model 3
  # MCOR: correlation matrix for the random effects
  # niter: number of iterations
  #
  require(ThreeWay)
  H=matrix(0,qqq,qqq^2)
  for (ii in 1:qqq){
    H[ii,(ii-1)*qqq+ii]=1
  }
  # data
  s=length(Kappa)
  x1=x2=cbind(dataset$X1,dataset$X2,dataset$X3,dataset$X4,dataset$X8,dataset$X15,dataset$X16)
  x3=cbind(dataset$X7,dataset$X9,dataset$X10,dataset$X11,dataset$X12,dataset$X14)
  y1=dataset$Y5
  y2=dataset$Y6
  y3=dataset$X15
  n=nrow(dataset)
  onesn=matrix(1,1,n) 
  
  #weigths
  w=array(runif(n*prod(Kappa)), c(n, Kappa))
  wi=w/(apply(w,c(1),sum))
  wi1=apply(wi, c(1,2), sum) 
  wi2=apply(wi, c(1,3), sum)
  wi3=apply(wi, c(1,4), sum)
  
  #set-up matrices and vectors 
  d1=matrix(1,Kappa[1],1)
  d2=matrix(1,Kappa[2],1)
  d3=matrix(1,Kappa[3],1) 
  y1m = d1%x%y1
  y2m = d2%x%y2
  y3m = d3%x%y3
  x1m = d1%x%x1
  x2m = d2%x%x2
  x3m = d3%x%x3
  z1 = diag(rep(1, Kappa[1]))%x%t(onesn)
  z2 = diag(rep(1, Kappa[2]))%x%t(onesn)
  z3 = diag(rep(1, Kappa[3]))%x%t(onesn)
  
  #conditional model 1
  mod1=glm(y1m~x1m+z1-1, family=poisson,weights=c(wi1))
  beta1 = mod1$coef
  mi1 = mod1$fitted 
  phi1 = dpois(y1m, mi1) 
  fi1 = matrix(ifelse(phi1 > 0, phi1, 1e-008), n, Kappa[1])
    
  #conditional model 2
  mod2=glm(y2m~x2m+z2-1, family=poisson,weights=c(wi2))
  beta2 = mod2$coef
  mi2 = mod2$fitted
  phi2 = dpois(y2m, mi2)
  fi2 = matrix(ifelse(phi2 > 0, phi2, 1e-008), n, Kappa[2])
  
  #conditional model 3
  mod3=glm(cbind(y3m,1-y3m)~x3m+z3-1, family=binomial,weights=c(wi3))
  beta3 = mod3$coef
  mi3 = mod3$fitted
  phi3 = dbinom(y3m, 1, mi3)
  fi3 = matrix(ifelse(phi3 > 0, phi3, 1e-008), n, Kappa[3])
    
  #joint density
  fi=array(NA, c(n, Kappa))  
  for (k1 in 1:Kappa[1])
  {
    for (k2 in 1:Kappa[2])
    {
      for (k3 in 1:Kappa[3])
      {
        fi[,k1,k2,k3]=fi1[,k1]*fi2[,k2]*fi3[,k3]
      }
    }
  }

  pik=apply(wi, c(2:(s+1)), sum)/n
  
  lik=fi*0
  for (i in 1:n)
  {
    lik[i,,,]=fi[i,,,]*pik
  }
  wi=lik/(apply(lik, c(1), sum)) 
  li=log(apply(lik, c(1), sum)) 
  likeli=sum(li)
  likeli2=likeli
  
  diff=1 
  niter=1
  
  #iterative cycle
  while(diff>1e-006)
  { 
    likeliold=likeli 
    likeli2old=likeli2
    wi1=apply(wi,c(1,2),sum) 
    wi2=apply(wi,c(1,3),sum)
    wi3=apply(wi,c(1,4),sum)
  
    #conditional model 1
    mod1=glm(y1m~x1m+z1-1,family=poisson,weights=c(wi1))
    beta1=mod1$coef
    mi1=mod1$fitted 
    phi1=dpois(y1m,mi1) 
    fi1=matrix(ifelse(phi1>0,phi1,1e-008),n,Kappa[1])
    
    #conditional model 2
    mod2=glm(y2m~x2m+z2-1,family=poisson,weights=c(wi2))
    beta2=mod2$coef
    mi2=mod2$fitted
    phi2=dpois(y2m,mi2)
    fi2=matrix(ifelse(phi2>0,phi2,1e-008),n,Kappa[2])
    
    #conditional model 3
    mod3=glm(cbind(y3m,1-y3m)~x3m+z3-1,family=binomial,weights=c(wi3))
    beta3=mod3$coef
    mi3=mod3$fitted
    phi3=dbinom(y3m,1,mi3)
    fi3=matrix(ifelse(phi3>0,phi3,1e-008),n,Kappa[3])
    
    #joint density
    for (k1 in 1:Kappa[1])
    {
      for (k2 in 1:Kappa[2])
      {
        for (k3 in 1:Kappa[3])
        {
          fi[,k1,k2,k3]=fi1[,k1]*fi2[,k2]*fi3[,k3]
        }
      }
    }
    pik=apply(wi,c(2:(s+1)),sum)/n
    
    #Parafac fitting
    pia=supermat(pik)$Xa
    sol=CPprob(pia,Kappa[1],Kappa[2],Kappa[3],qqq,1e-9,50,10)
    piae=sol$Xe
    pie=rarray(piae,Kappa[1],Kappa[2],Kappa[3])

    lik2=fi*0
    for (i in 1:n)
    {
      lik2[i,,,]=fi[i,,,]*pie
    }
    li2=log(apply(lik2,c(1),sum)) 
    likeli2=sum(li2)
    wi2=lik2/(apply(lik2,c(1),sum)) 
           
    lik=fi*0
    for (i in 1:n)
    {
      lik[i,,,]=fi[i,,,]*pik
    }
    li=log(apply(lik,c(1),sum)) 
    likeli=sum(li)
    wi=lik/(apply(lik,c(1),sum)) 
    diff=(likeli-likeliold)/abs(likeliold)
    diff2=(likeli2-likeli2old)/abs(likeli2old)
    if (diff2<0){break}
    niter=niter+1 
  }
  
  #parameter estimates
  SS2=CalSigma2(beta1, beta2, beta3, Kappa[1], Kappa[2], Kappa[3], pie)
  betag=c(beta1[1], beta2[1],beta3[1]) 
  ug=c(beta1[2:length(beta1)], beta2[2:length(beta2)],beta3[2:length(beta3)])
  if (is.null(ncol(x1))==F)
  {
    betag=c(beta1[1:(ncol(x1))],beta2[1:(ncol(x2))],beta3[1:(ncol(x3))]) 
    ug=c(beta1[(ncol(x1)+1):length(beta1)],beta2[(ncol(x2)+1):length(beta2)],beta3[(ncol(x3)+1):length(beta3)])
  }
  
  #Results
  resul=list() 
  resul$beta0=SS2$beta0
  resul$beta=betag 
  resul$K1=Kappa[1]
  resul$K2=Kappa[2]
  resul$K3=Kappa[3]  
  resul$u=ug 
  resul$pi=pik
  resul$pirid=pie
  resul$MCOV=c(SS2$cov)
  resul$post=wi
  resul$likeli=likeli2
  resul$Aparafac=sol$A
  resul$Bparafac=sol$B
  resul$Cparafac=sol$C
  resul$Hparafac=sol$HA
  resul$mod1=mod1
  resul$mod2=mod2
  resul$mod3=mod3
  resul$MCOR=SS2$cor
  resul$niter=niter
  return(resul)
}

EMdataparafacseq=function(dataset,Kappa,qqq) 
{
  #
  # Parafac approach: fitting function (sequential procedure)
  #
  # Input:
  # dataset: data.frame
  # Kappa: vector with the number of locations in each profile
  # qqq: number of components in Parafac
  #
  # Output:
  # beta0: intercepts (used for computing MCOV and MCOR) 
  # beta: parameter estimates
  # K1: number of locations in profile 1 
  # K2: number of locations in profile 2
  # K3: number of locations in profile 3
  # u: estimated intercepts (locations)
  # pi: estimated joint probabilities (tensor)
  # pirid: Parafac approximation of the joint probabilities (tensor)
  # MCOV: covariance matrix for the random effects
  # post: posterior probabilities
  # likeli: likelihood value upon convergence
  # Aparafac: estimated matrix P1
  # Bparafac: estimated matrix P2
  # Cparafac: estimated matrix P3
  # Hparafac: estimated matrix T
  # mod1: conditional model 1
  # mod2: conditional model 2
  # mod3: conditional model 3
  # MCOR: correlation matrix for the random effects
  # niter: number of iterations
  #
  require(ThreeWay)
  H=matrix(0,qqq,qqq^2)
  for (ii in 1:qqq){
    H[ii,(ii-1)*qqq+ii]=1
  }
  
  #data
  s=length(Kappa)
  x1=x2=cbind(dataset$X1,dataset$X2,dataset$X3,dataset$X4,dataset$X8,dataset$X15,dataset$X16)
  x3=cbind(dataset$X7,dataset$X9,dataset$X10,dataset$X11,dataset$X12,dataset$X14)  
  y1=dataset$Y5
  y2=dataset$Y6
  y3=dataset$X15
  n=nrow(dataset)
  onesn=matrix(1,1,n) 
  
  #weights 
  w=array(runif(n*prod(Kappa)),c(n,Kappa))
  wi=w/(apply(w,c(1),sum))
  wi1=apply(wi,c(1,2),sum) 
  wi2=apply(wi,c(1,3),sum)
  wi3=apply(wi,c(1,4),sum)

  #set-up matrices and vectors 
  d1=matrix(1,Kappa[1],1)
  d2=matrix(1,Kappa[2],1)
  d3=matrix(1,Kappa[3],1) 
  y1m=d1%x%y1
  y2m=d2%x%y2
  y3m=d3%x%y3
  x1m=d1%x%x1
  x2m=d2%x%x2
  x3m=d3%x%x3
  z1=diag(rep(1,Kappa[1]))%x%t(onesn)
  z2=diag(rep(1,Kappa[2]))%x%t(onesn)
  z3=diag(rep(1,Kappa[3]))%x%t(onesn)
  
  #conditional model 1
  mod1=glm(y1m~x1m+z1-1,family=poisson,weights=c(wi1))
  beta1=mod1$coef
  mi1=mod1$fitted 
  phi1=dpois(y1m,mi1) 
  fi1=matrix(ifelse(phi1>0,phi1,1e-008),n,Kappa[1])
  
  #conditional model 2
  mod2=glm(y2m~x2m+z2-1,family=poisson,weights=c(wi2))
  beta2=mod2$coef
  mi2=mod2$fitted
  phi2=dpois(y2m,mi2)
  fi2=matrix(ifelse(phi2>0,phi2,1e-008),n,Kappa[2])
  
  #conditional model 3
  mod3=glm(cbind(y3m,1-y3m)~x3m+z3-1,family=binomial,weights=c(wi3))
  beta3=mod3$coef
  mi3=mod3$fitted
  phi3=dbinom(y3m, 1, mi3)
  fi3=matrix(ifelse(phi3>0,phi3,1e-008),n,Kappa[3])
  
  #joint density
  fi=array(NA,c(n,Kappa))  
  for (k1 in 1:Kappa[1])
  {
    for (k2 in 1:Kappa[2])
    {
      for (k3 in 1:Kappa[3])
      {
        fi[,k1,k2,k3]=fi1[,k1]*fi2[,k2]*fi3[,k3]
      }
    }
  }
  pik=apply(wi, c(2:(s+1)), sum)/n
  lik=fi*0
  for (i in 1:n)
  {
    lik[i,,,]=fi[i,,,]*pik
  }
  wi=lik/(apply(lik, c(1), sum)) 
  li=log(apply(lik, c(1), sum)) 
  likeli=sum(li)
  likeli2=likeli
  
  #iterative cycle
  diff=1 
  niter=1
  while(diff>1e-006)
  { 
    likeliold=likeli 
    likeli2old=likeli2
    wi1=apply(wi,c(1,2),sum) 
    wi2=apply(wi,c(1,3),sum)
    wi3=apply(wi,c(1,4),sum)

    #conditional model 1
    mod1=glm(y1m~x1m+z1-1,family=poisson,weights=c(wi1))
    beta1=mod1$coef
    mi1=mod1$fitted 
    phi1=dpois(y1m,mi1) 
    fi1=matrix(ifelse(phi1>0,phi1,1e-008),n,Kappa[1])
    
    #conditional model 2
    mod2=glm(y2m~x2m+z2-1,family=poisson,weights=c(wi2))
    beta2=mod2$coef
    mi2=mod2$fitted
    phi2=dpois(y2m,mi2)
    fi2=matrix(ifelse(phi2>0,phi2,1e-008),n,Kappa[2])
    
    #conditional model 3
    mod3=glm(cbind(y3m,1-y3m)~x3m+z3-1,family=binomial,weights=c(wi3))
    beta3=mod3$coef
    mi3=mod3$fitted
    phi3=dbinom(y3m,1,mi3)
    fi3=matrix(ifelse(phi3>0,phi3,1e-008),n,Kappa[3])
    
    #joint density
    for (k1 in 1:Kappa[1])
    {
      for (k2 in 1:Kappa[2])
      {
        for (k3 in 1:Kappa[3])
        {
          fi[,k1,k2,k3]=fi1[,k1]*fi2[,k2]*fi3[,k3]
        }
      }
    }
    pik=apply(wi,c(2:(s+1)),sum)/n
    pie=pik
    lik2=fi*0
    for (i in 1:n)
    {
      lik2[i,,,]=fi[i,,,]*pie
    }
    li2=log(apply(lik2,c(1),sum)) 
    likeli2=sum(li2)
    wi2=lik2/(apply(lik2,c(1),sum)) 
    lik=fi*0
    for (i in 1:n)
    {
      lik[i,,,]=fi[i,,,]*pik
    }
    li=log(apply(lik,c(1),sum)) 
    likeli=sum(li)
    wi=lik/(apply(lik,c(1),sum)) 
    diff=(likeli-likeliold)/abs(likeliold)
    diff2=(likeli2-likeli2old)/abs(likeli2old)
    if (diff2<0){break}
    niter=niter+1 
  }

  #Parafac fitting
  pia=supermat(pik)$Xa
  sol=CPprob(pia,Kappa[1],Kappa[2],Kappa[3],qqq,1e-6,999,10)
  piae=sol$Xe
  pie=rarray(piae,Kappa[1],Kappa[2],Kappa[3])

  lik2=fi*0
  for (i in 1:n)
  {
    lik2[i,,,]=fi[i,,,]*pie
  }
  li2=log(apply(lik2,c(1),sum)) 
  likeli2=sum(li2)
  wi2=lik2/(apply(lik2,c(1),sum)) 
  
  lik=fi*0
  for (i in 1:n)
  {
    lik[i,,,]=fi[i,,,]*pik
  }
  li=log(apply(lik,c(1),sum)) 
  likeli=sum(li)
  wi=lik/(apply(lik,c(1),sum)) 
  
  #parameter estimates
  SS2=CalSigma2(beta1,beta2,beta3,Kappa[1],Kappa[2],Kappa[3],pie)
  betag=c(beta1[1], beta2[1],beta3[1]) 
  ug=c(beta1[2:length(beta1)],beta2[2:length(beta2)],beta3[2:length(beta3)])
  if (is.null(ncol(x1))==F)
  {
    betag=c(beta1[1:(ncol(x1))],beta2[1:(ncol(x2))],beta3[1:(ncol(x3))]) 
    ug=c(beta1[(ncol(x1)+1):length(beta1)],beta2[(ncol(x2)+1):length(beta2)],beta3[(ncol(x3)+1):length(beta3)])
  }
  
  #Results
  resul=list() 
  resul$beta0=SS2$beta0
  resul$beta=betag 
  resul$K1=Kappa[1]
  resul$K2=Kappa[2]
  resul$K3=Kappa[3]  
  resul$u=ug 
  resul$pi=pik
  resul$pirid=pie
  resul$MCOV=c(SS2$cov)
  resul$post=wi
  resul$likeli=likeli2
  resul$Aparafac=sol$A
  resul$Bparafac=sol$B
  resul$Cparafac=sol$C
  resul$Hparafac=sol$HA
  resul$mod1=mod1
  resul$mod2=mod2
  resul$mod3=mod3
  resul$MCOR=SS2$cor
  resul$niter=niter
  return(resul)
}

CPprob <- function(XA,n,m,p,r,conv,maxit,rs)
{
  # inner function in EMdataparafac and EMdataparafacseq
  require(lsei)
  require(MASS)
  require(ThreeWay)
  sumneg=function(x) sum(x[x<0])
  sumpos=function(x) sum(x[x>0])
  XA=as.matrix(XA)
  XB=permnew(XA,n,m,p)
  XC=permnew(XB,m,p,n)
  ssx=sum(XA^2)
  fopt=ssx*10^6
  HA=matrix(0,r,r^2)
  for (s in 1:r)
  {
    HA[s,(s-1)*r+s]=1/r
  }
  HB=permnew(HA,r,r,r)
  HC=permnew(HB,r,r,r)
  cf=c()
  #first random start based on Parafac solution (max 100 iterations) with estimates of A,B,C in [0,1] and summing to 1
  for (nrs in 1:rs)
  {
    if (nrs==1)
    {
      sol=CPfuncrep(XA,n,m,p,r,1,1,1,0,conv,100)
      A=sol$A
      B=sol$B
      C=sol$C
      for (s in 1:r)
      {
        if (sumpos(A[,s])<abs(sumneg(A[,s]))) A[,s]=-A[,s]
        if (sumpos(B[,s])<abs(sumneg(B[,s]))) B[,s]=-B[,s]
        if (sumpos(C[,s])<abs(sumneg(C[,s]))) C[,s]=-C[,s]
      }
      A[A<0]=0
      A[A>1]=1
      sA=apply(A,2,sum)
      A=A%*%diag(1/sA,nrow=length(sA))
      B[B<0]=0
      B[B>1]=1
      sB=apply(B,2,sum)
      B=B%*%diag(1/sB,nrow=length(sB))
      C[C<0]=0
      C[C>1]=1
      sC=apply(C,2,sum)
      C=C%*%diag(1/sC,nrow=length(sC))
    }
    else{
      if (n>=r)
      {
        A=orth(matrix(runif(n*r,0,1),nrow=n)-0.5)
        A=matrix(A,ncol=r)
      }else{
        A=orth(matrix(runif(r*r,0,1),nrow=r)-0.5)
        A=A[1:n,]
      }
      for (s in 1:r)
      {
        if (sumpos(A[,s])<abs(sumneg(A[,s]))) A[,s]=-A[,s]
      }
      A[A<0]=0
      A[A>1]=1
      sA=apply(A,2,sum)
      A=A%*%diag(1/sA,nrow=length(sA))
      if (m>=r)
      {
        B=orth(matrix(runif(m*r,0,1),nrow=m)-0.5)
        B=matrix(B,ncol=r)
      }else{
        B=orth(matrix(runif(r*r,0,1),nrow=r)-0.5)
        B=B[1:m,]
      }
      for (s in 1:r)
      {
        if (sumpos(B[,s])<abs(sumneg(B[,s]))) B[,s]=-B[,s]
      }
      B[B<0]=0
      B[B>1]=1
      sB=apply(B,2,sum)
      B=B%*%diag(1/sB,nrow=length(sB))
      if (p>=r)
      {
        C=orth(matrix(runif(p*r,0,1),nrow=p)-0.5)
        C=matrix(C,ncol=r)
      }else{
        C=orth(matrix(runif(r*r,0,1),nrow=r)-0.5)
        C=C[1:p,]
      }
      for (s in 1:r)
      {
        if (sumpos(C[,s])<abs(sumneg(C[,s]))) C[,s]=-C[,s]
      }
      C[C<0]=0
      C[C>1]=1
      sC=apply(C,2,sum)
      C=C%*%diag(1/sC,nrow=length(sC))
    }
    
    #iterative cycle
    f=sum((XA-A%*%HA%*%(t(C)%x%t(B)))^2)
    fold=f+2*conv*f
    iter=0
    while ((fold-f>conv*f) & (iter<maxit))
    {
      fold=f
    
      #row-wise update of A
      F=t(HA%*%(t(C)%x%t(B)))
      if (r>1)
      {
        for (s in 1:r)
        {
          As=matrix(A[,-s],ncol=r-1)
          Fs=matrix(F[,-s],ncol=r-1)
          f=matrix(F[,s],ncol=1)
          Y=XA-As%*%t(Fs)
          old=A[,s]
          A[,s]=updateCOL(matrix(as.vector(Y),ncol=1),kronecker(f,diag(1,nrow=n)),n,old)
        }
      }else{
        old=A
        A=updateCOL(matrix(as.vector(XA),ncol=1),kronecker(F,diag(1,nrow=n)),n,old)
      }
      
      #row-wise update of B
      F=t(HB%*%(t(A)%x%t(C)))
      if (r>1)
      {
        for (s in 1:r)
        {
          Bs=matrix(B[,-s],ncol=r-1)
          Fs=matrix(F[,-s],ncol=r-1)
          f=matrix(F[,s],ncol=1)
          Y=XB-Bs%*%t(Fs)
          old=B[,s]
          B[,s]=updateCOL(matrix(as.vector(Y),ncol=1),kronecker(f,diag(1,nrow=m)),m,old);
        }
      }else{
        old=B
        B=updateCOL(matrix(as.vector(XA),ncol=1),kronecker(F,diag(1,nrow=m)),m,old)
      }
      
      #row-wise update of C
      F=t(HC%*%(t(B)%x%t(A)))
      if (r>1)
      {
        for (s in 1:r)
        {
          Cs=matrix(C[,-s],ncol=r-1)
          Fs=matrix(F[,-s],ncol=r-1)
          f=matrix(F[,s],ncol=1)
          Y=XC-Cs%*%t(Fs)
          old=C[,s]
          C[,s]=updateCOL(matrix(as.vector(Y),ncol=1),kronecker(f,diag(1,nrow=p)),p,old)
        }
      }else{
        old=C
        C=updateCOL(matrix(as.vector(XA),ncol=1),kronecker(F,diag(1,nrow=p)),p,old)
      }
      
      # update of H
      Hred=matrix(0,nrow=n*m*p,ncol=r)
      old=vector()
      for (s in 1:r){
        Hred[,s]=kronecker(C[,s],kronecker(B[,s],A[,s]))
        old[s]=HA[s,(s-1)*r+s]
      }
      h=updateCOL(matrix(as.vector(XA),ncol=1),Hred,r,old)
      for (s in 1:r){
        HA[s,(s-1)*r+s]=h[s]
      }
      HB=permnew(HA,r,r,r)
      HC=permnew(HB,r,r,r)
      
      f=sum((XA-A%*%HA%*%(t(C)%x%t(B)))^2)
      iter=iter+1
    }
    cf[nrs]=f
    if (f<fopt)
    {
      fopt=f
      Aopt=A
      Bopt=B
      Copt=C
      Hopt=HA
      Xeopt=A%*%HA%*%(t(C)%x%t(B))
    }
  }
  fp=(1-fopt/ssx)*100
  tripcos=min(phi(Aopt,Aopt)*phi(Bopt,Bopt)*phi(Copt,Copt))
  names(tripcos)=c("Minimal triple cosine")
  out=list()
  out$A=Aopt
  out$B=Bopt
  out$C=Copt
  out$HA=Hopt
  out$f=cf
  out$fopt=fopt
  out$fp=fp
  out$Xe=Xeopt
  out$tripcos=tripcos
  return(out)
}

CalSigma2=function(par1, par2, par3, K1, K2, K3, gpi)
{
  # inner function in EMdataparafac and EMdataparafacseq  
  S=length(dim(gpi))
  Beta0=matrix(0,1,S)
  Sigma=matrix(0,S,S)
  KK=max(K1,K2,K3)
  Kappa=c(K1,K2,K3)
  U=matrix(0,KK,S)
  
  beta1=par1[1:(length(par1)-K1)]
  beta2=par2[1:(length(par2)-K2)]
  beta3=par3[1:(length(par3)-K3)]
  Beta=c(beta1,beta2,beta3)
  u1=par1[(length(beta1)+1):length(par1)]
  u2=par2[(length(beta2)+1):length(par2)]
  u3=par3[(length(beta3)+1):length(par3)]
  U=cbind(u1,u2,u3)
  ScartoU=U*0
  
  for (j in 1:S)
  {
    Beta0[j]=sum(U[(1:Kappa[j]),j]*apply(gpi,c(j),sum))
    ScartoU[(1:Kappa[j]),j]=U[(1:Kappa[j]),j]-Beta0[j]
    Sigma[j,j]=sum((ScartoU[(1:Kappa[j]),j])^2*apply(gpi,c(j),sum))
  }
  for (j in 1:(S-1))
  {
    for (jj in (j+1):S)
    {
      Sigma[j,jj]=Sigma[jj,j]=t(ScartoU[(1:Kappa[j]),j])%*%apply(gpi,c(j,jj),sum)%*%(ScartoU[(1:Kappa[jj]),jj])
    }
  }
  res=list()
  res$beta=Beta
  res$beta0=Beta0
  res$cov=Sigma
  res$cor=solve(diag(sqrt(diag(Sigma))))%*%Sigma%*%solve(diag(sqrt(diag(Sigma))))
  return(res)
}

updateCOL <- function(y,X,K,old)
{
  # inner function in CPprob
  require(MASS,lsei)
  Z=matrix(Null(rep(1,K)),nrow=K)
  f=y-X%*%matrix(rep(1,K),nrow=K)/K
  E=X%*%Z
  if (sum(E^2)>0)
  {
    mp=lsei(E,f,e=Z,f=matrix(rep(-1,K),nrow=K)/K)
    sol=Z%*%mp+matrix(rep(1,K),nrow=K)/K
  }else{
    sol=old
  }
  return(sol)
}