#################################################################################################
# Graph #########################################################################################
#################################################################################################
ac <- list(pad1=0.5, pad2=0.5, tck=0.5)
mycol <- gray.colors(n=5)
ps <- list(box.rectangle=list(col=1, fill=c("gray70")),
           box.umbrella=list(col=1, lty=1),
           dot.symbol=list(col=1),
           dot.line=list(col=1, lty=3),
           plot.symbol=list(col=1, cex=0.7),
           plot.line=list(col=1),
           plot.polygon=list(col="gray80"),
           superpose.line=list(col=mycol),
           superpose.symbol=list(col=mycol),
           superpose.polygon=list(col=mycol),
           strip.background=list(col=c("gray90","gray70")),
           layout.widths=list(
               left.padding=0.25,
               right.padding=-1,
               ylab.axis.padding=0),
           layout.heights=list(
               bottom.padding=0.25,
               top.padding=0,
               axis.xlab.padding=0,
               xlab.top=0),
           axis.components=list(bottom=ac, top=ac, left=ac, right=ac)
)
panel.segplotBy <- function(x, y, z, centers, subscripts, groups, f, ...){
    d <- 2*((as.numeric(groups)-1)/(nlevels(groups)-1))-1
    z <- as.numeric(z)+f*d
    panel.segplot(x, y, z, centers=centers,
                  subscripts=subscripts, ...)
}

panel.cbH <- function(x, y, ly, uy,
                      subscripts, cty,
                      col.line = plot.line$col,
                      lwd = plot.line$lwd,
                      desloc = NULL,
                      fill = 1, alpha = 0.1, length = 0.05, ...) {
    plot.line <- trellis.par.get("plot.line")
    if (is.null(desloc)) {
        desloc <- rep(0, length(uy))
    }
    y <- as.numeric(y)
    x <- as.numeric(x)
    or <- order(x)
    ly <- as.numeric(ly[subscripts])
    uy <- as.numeric(uy[subscripts])
    xo <- x[or]
    yo <- y[or]
    lyo <- ly[or]
    uyo <- uy[or]
    desl <- desloc[subscripts]
    if (cty == "bands") {
        panel.polygon(c(xo, rev(xo)), c(lyo, rev(uyo)), col = fill,
                      alpha = alpha, border = NA)
        panel.lines(xo, lyo, lty = 3, lwd = 0.5, col = col.line)
        panel.lines(xo, uyo, lty = 3, lwd = 0.5, col = col.line)
    }
    if (cty == "bars") {
        panel.arrows(xo + desl, lyo, xo + desl, uyo, length = length,
                     code = 3, angle = 90, col = col.line, lwd = lwd)
    }
    panel.xyplot(x + desl, y, subscripts = subscripts,
                 col.line = col.line, lwd = lwd, ...)
}


prepanel.cbH <- function(y, ly, uy, subscripts) {
    ly <- as.numeric(ly[subscripts])
    uy <- as.numeric(uy[subscripts])
    y <- as.numeric(y[subscripts])
    list(ylim = range(y, uy, ly, finite = TRUE))
}
#' @name centfac
#' @export
#' @author Walmes Zeviani, \email{walmes@@ufpr.br}.
#' @title Numeric Centered Factor Levels
#'
#' @description This function receveis a factor and return a numeric
#'     vector with equally spaced factor levels centered at 0.
#'
#' @param group A factor.
#' @param space A numeric value to be used as the space between
#'     levels. If \code{NULL}, the space is determined by the
#'     \code{group}.
#' @examples
#'
#' centfac(warpbreaks$tension)
#' centfac(warpbreaks$tension, space = 1)
#' centfac(warpbreaks$wool)
#' centfac(warpbreaks$wool, space = 1)
#'
centfac <- function(group, space = NULL) {
    stopifnot(is.factor(group))
    if (is.null(space)) {
        space <- 0.5/nlevels(group)
    }
    d <- 2 * ((as.integer(group) - 1)/(nlevels(group) - 1)) - 1
    return(space * d)
}

#' @name panel.groups.segplot
#' @export
#' @author Walmes Zeviani, \email{walmes@@ufpr.br}.
#' @title Panel to Plot Confidence Intervals by Groups in
#'     \code{segplot()}
#'
#' @description This function allows non overlapping error bars in
#'     \code{latticeExtra::segplot()}. It has the \code{groups =}
#'     argument.
#'
#' @param x,y,z,centers,data,subscripts,... Arguments passed to
#'     \code{\link[latticeExtra]{segplot}()}.
#'
#' @param groups The grouping variable (\code{factor}), with the same
#'     length of \code{lwr} e \code{upr}.
#'
#' @param gap Scalar that is the distance among segments. Default is
#'     0.1. If the grouping factor has \eqn{k} levels, so \eqn{0 \leq
#'     \textrm{gap} < 1/k}. A negative value for \code{gap} will put the
#'     segments in a reversed order.
#'
#' @seealso \code{\link[latticeExtra]{segplot}()}.
#' @import latticeExtra
#' @examples
#'
#' library(latticeExtra)
#'
#' m0 <- lm(log(breaks) ~ wool * tension, data = warpbreaks)
#'
#' pred <- with(warpbreaks, expand.grid(KEEP.OUT.ATTRS = TRUE,
#'                                      wool = levels(wool),
#'                                      tension = levels(tension)))
#'
#' pred <- cbind(pred,
#'               predict(m0, newdata = pred, interval = "confidence"))
#' str(pred)
#'
#' segplot(wool ~ lwr + upr, centers = fit, data = pred,
#'         draw = FALSE, horizontal = FALSE)
#'
#' segplot(wool ~ lwr + upr, centers = fit, data = pred,
#'         draw = FALSE, horizontal = FALSE,
#'         groups = tension, gap = 0.05,
#'         panel = panel.groups.segplot)
#'
panel.groups.segplot <- function(x, y, z, centers,
                                 groups, gap = NULL,
                                 data, subscripts, ...) {
    if (!missing(data)) {
        data <- eval(data, envir = parent.frame())
        groups <- data[, deparse(substitute(groups))]
    }
    stopifnot(is.factor(groups))
    stopifnot(length(groups) == length(z))
    z <- as.numeric(z) + centfac(groups, space = gap)
    panel.segplot(x, y, z, centers = centers,
                  subscripts = subscripts, ...)
}

panel.beeswarm <- function(x, y, subscripts, spread, ...) {
    xx <- x
    yy <- y
    aux <- by(cbind(yy, xx, subscripts), xx, function(i) {
        or <- order(i[, 1])
        ys <- i[or, 1]
        yt <- table(ys)
        dv <- sapply(unlist(yt),
                     FUN = function(j) {
                         seq(1, j, l = j) - (j + 1)/2
                     })
        if (!is.list(dv)) {
            dv <- as.list(dv)
        }
        xs <- i[or, 2] + spread * do.call(c, dv)
        cbind(x = xs, y = ys, subscripts[or])
    })
    aux <- do.call(rbind, aux)
    panel.xyplot(aux[, 1], aux[, 2], subscripts = aux[, 3], ...)
}

#' @name apc
#' @author Walmes Zeviani, \email{walmes@@ufpr.br}.
#' @export
#' @title Generate Matrix of All Pairwise Comparisons (Tukey contrasts)
#' @description This function takes a matrix where each line defines a
#'     linear function of the parameters to estimate a marginal mean
#'     (aka least squares mean) and return the matrix that define the
#'     contrasts among these means. All pairwise contrasts are returned
#'     (aka Tukey contrasts). The matrix with these contrasts can be
#'     passed to \code{\link[multcomp]{glht}()} to estimate them or used
#'     in explicit matricial calculus.
#' @param lfm a \eqn{k \times p} matrix where each line defines a linear
#'     function to estimate a lsmean. In general, these matrices are
#'     obtained by using \code{\link[doBy]{LSmatrix}()}.
#' @param lev a character vector with length equals to the numbers of
#'     lines of \code{lfm} matrix, (\eqn{k}). Default is \code{NULL} and
#'     the row names of code{lfm} is used. If row names is also
#'     \code{NULL}, incremental integer values are used to identify the
#'     comparisons.
#' @return a \eqn{K\times p} matrix with the linear functions that
#'     define all pairwise contrasts. \eqn{K} is \eqn{{k}\choose{2}}.
#' @seealso \code{\link{apmc}()}, \code{\link[doBy]{LSmatrix}()}.
#' @examples
#'
#' X <- diag(3)
#' rownames(X)
#' apc(X)
#'
#' rownames(X) <- letters[nrow(X):1]
#' apc(X)
#'
#' apc(X, lev = LETTERS[1:nrow(X)])
#'
#' # Objects from doBy::LSmatrix() have an "grid" attribute.
#' attr(X, "grid") <- data.frame(n = LETTERS[1:nrow(X)])
#' rownames(X) <- NULL
#' apc(X)
#'
apc <- function(lfm, lev = NULL) {
    nlev <- nrow(lfm)
    rn <- rownames(lfm)
    a <- attr(lfm, "grid")
    if (is.null(lev)) {
        if (!is.null(a)) {
            lev <- apply(a, 1, paste, collapse = ":")
        } else if (!is.null(rn)) {
            lev <- rn
        } else {
            lev <- as.character(1:nlev)
        }
    }
    cbn <- utils::combn(seq_along(lev), 2)
    M <- lfm[cbn[1, ], ] - lfm[cbn[2, ], ]
    if (is.vector(M)) {
        dim(M) <- c(1, length(M))
    }
    rownames(M) <- paste(lev[cbn[1, ]], lev[cbn[2, ]], sep = "-")
    return(M)
}

#' @name apmc
#' @author Walmes Zeviani, \email{walmes@@ufpr.br}.
#' @export
#' @title A Wraper of glht to Get All Pairwise Mean Comparisons
#' @description This function performs all pairwise compararisons among
#'     means returning pontual and intervalar estimates followed by
#'     letters to easy discriminate values. It is in fact a wraper of
#'     \code{\link[multcomp]{glht}()}.
#' @param X a matrix where each line is a linear function of the model
#'     parameters to estimate a least squares mean. In most pratical
#'     cases, it is an object from the \code{\link[doBy]{LSmatrix}()}.
#' @param model a model with class recognized by
#'     \code{\link[multcomp]{glht}()}.
#' @param focus a string with the name of the factor which levels are
#'     being compared.
#' @param test a p-value correction method. See
#'     \code{\link[stats]{p.adjust.methods}()}.
#' @param level the experimentwise significance level for the multiple
#'     comparisons. The individual coverage of the confidence interval
#'     is \code{1-level}. Default is 0.05.
#' @param cld2 Logical, if \code{TRUE} uses the \code{\link{cld2}()}
#'     functions, otherwise uses the \code{\link[multcomp]{cld}()}
#'     function.
#' @return a \code{data.frame} with interval estimates and compact
#'     letter display for the means comparisons.
#' @seealso \code{\link{apc}()}, \code{\link[doBy]{LSmatrix}()},
#'     \code{\link[multcomp]{glht}()}.
#' @examples
#'
#' library(doBy)
#' library(multcomp)
#'
#' # Single factor.
#' m0 <- lm(weight ~ feed, data = chickwts)
#' anova(m0)
#'
#' # Prepare the matrix to estimate lsmeans.
#' L <- LSmatrix(m0, effect = "feed")
#' rownames(L) <- levels(chickwts$feed)
#' apmc(L, model = m0, focus = "feed", test = "fdr")
#'
#' data(warpbreaks)
#'
#' # Two factors (complete factorial).
#' m1 <- lm(breaks ~ wool * tension, data = warpbreaks)
#' anova(m1)
#'
#' L <- LSmatrix(m1, effect = c("wool", "tension"))
#' attributes(L)
#'
#' Ls <- by(L, INDICES = attr(L, "grid")$tension, FUN = as.matrix)
#' Ls <- lapply(Ls, "rownames<-", levels(warpbreaks$wool))
#'
#' # Comparing means of wool in each tension.
#' lapply(Ls, apmc, model = m1, focus = "wool",
#'        test = "single-step", level = 0.1)
#'
#' # Two factors (incomplete factorial).
#' warpbreaks <- subset(warpbreaks, !(tension == "H" & wool == "A"))
#' xtabs(~tension + wool, data = warpbreaks)
#'
#' # There is NA in the estimated parameters.
#' m2 <- lm(breaks ~ wool * tension, data = warpbreaks)
#' coef(m2)
#'
#' X <- model.matrix(m2)
#' b <- coef(m2)
#'
#' X <- X[, !is.na(b)]
#'
#' # unique(X)
#'
#' # Uses the full estimable model matriz.
#' m3 <- update(m2, . ~ 0 + X)
#'
#' # These models are in fact the same.
#' anova(m2, m3)
#'
#' # LS matrix has all cells.
#' L <- LSmatrix(m2, effect = c("wool", "tension"))
#' g <- attr(L, "grid")
#' L <- L[, !is.na(b)]
#' i <- 5
#' L <- L[-i, ]
#' g <- g[-i, ]
#'
#' rownames(L) <- g$tension
#' Ls <- by(L, INDICES = g$wool, FUN = as.matrix)
#'
#' # LSmeans with MCP test.
#' lapply(Ls, apmc, model = m3, focus = "tension",
#'        test = "single-step", level = 0.1, cld2 = TRUE)
#'
#' # Sample means.
#' aggregate(breaks ~ tension + wool, data = warpbreaks, FUN = mean)
#'
apmc <- function(X, model, focus, test = "single-step", level = 0.05,
                 cld2 = FALSE) {
    if (is.null(rownames(X))) {
        stop("The X matrix must have row names.")
    }
    Xc <- apc(X)
    g <- multcomp::glht(model, linfct = X)
    ci <- confint(g, level = 1 - level,
                  calpha = multcomp::univariate_calpha())$confint
    ci <- as.data.frame(ci)
    names(ci) <- tolower(names(ci))
    names(ci)[1] <- "fit"
    h <- summary(multcomp::glht(model, linfct = Xc),
                 test = adjusted(type = test))
    h$type <- "Tukey"
    h$focus <- focus
    if (cld2) {
        ci$cld <- cld2(h,
                       level = level)$mcletters$Letters
        # ci$cld <- ordered_cld(ci$cld, ci$fit)
    } else {
        ci$cld <- multcomp::cld(h, level = level,
                                decreasing = TRUE)$mcletters$Letters
    }
    ci <- cbind(rownames(ci), ci)
    names(ci)[1] <- focus
    rownames(ci) <- NULL
    return(ci)
}

#' @name cld2
#' @author Walmes Zeviani, \email{walmes@@ufpr.r}.
#' @export
#' @title Modified Compact Letter Display to Irregular Designs
#' @description This functions get the compact letter display for
#'     objects of class \code{"glht"}. Modification was done to get the
#'     letters to design with missing cells, non completelly crossed
#'     factorial designs and nested factorial designs. These models are
#'     usually declared by a model matrix to have all effects
#'     estimated. It is assumed that Tukey contrasts was used.
#' @param object an object returned by \code{\link[multcomp]{glht}()}. It
#'     is assumed that the matrix used as the \code{linfct} argument in
#'     \code{glht} corresponds to a matrix to get Tukey contrasts of
#'     least squares means.
#' @param level the nominal significance level.
#' @return an object of class \code{"cld"} with letters to resume mean
#'     comparisons.
#' @seealso \code{\link{apc}()}, \code{\link[doBy]{LSmatrix}()},
#'     \code{\link[multcomp]{glht}()}.
#' @import multcomp
#' @examples
#'
#' # Toy data 1: experiment with cultivars in several locations.
#' td1 <- expand.grid(loc = gl(5, 1),
#'                    block = gl(3, 1),
#'                    cult = LETTERS[1:6])
#' td1 <- subset(td1, !(loc == 1 & cult == "A"))
#' td1 <- subset(td1, !(loc == 2 & cult == "B"))
#' xtabs(~loc + cult, td1)
#' td1$y <- seq_len(nrow(td1))
#'
#' library(lme4)
#'
#' # Fit the mixed model.
#' m0 <- lmer(y ~ loc * cult + (1 | loc:block), data = td1)
#' logLik(m0)
#'
#' # The same model but without rank deficience.
#' td1$loccult <- with(td1, interaction(loc, cult, drop = TRUE))
#' m1 <- lmer(y ~ loccult + (1 | loc:block), data = td1)
#' logLik(m1)
#'
#' library(doBy)
#'
#' X <- LSmatrix(lm(nobars(formula(m1)), data = td1), effect = "loccult")
#' rownames(X) <- levels(td1$loccult)
#' dim(X)
#'
#' Xs <- X[grepl(x = rownames(X), "^1\\."),]
#' Xc <- apc(Xs)
#'
#' library(multcomp)
#'
#' g <- summary(glht(m1, linfct = Xc), test = adjusted(type = "fdr"))
#'
#' cld2(g)
#'
#' confint(glht(m1, linfct = Xs), calpha = univariate_calpha())
#'
cld2 <- function(object, level = 0.05) {
    lvl_order <- unique(unlist(
        strsplit(rownames(object$linfct), "-")))
    signif <- (object$test$pvalues < level)
    ret <- list()
    ret$signif <- signif
    ret$comps <- do.call(rbind,
                         strsplit(rownames(object$linfct), "-"))
    ret$mcletters <-
        insert_absorb(x = signif,
                      decreasing = TRUE,
                      comps = ret$comps,
                      lvl_order = lvl_order)
    class(ret) <- "cld"
    return(ret)
}

#' @name ordered_cld
#' @author Walmes Zeviani, \email{walmes@@ufpr.r}.
#' @export
#' @title Order Letters According to Numeric Vector
#' @description This function order the letters in the compact letter
#'     display to the highest estimate receive the letter \code{a}. This
#'     is a convetion in most software for analysis of experiments.
#' @param let Character vector with the letters returned by
#'     \code{\link[multcomp]{cld}()} or \code{\link{cld2}()}.
#' @param means Numeric vector with the corresponding estimates in which
#'     the highest value will have the letter \code{a}.
#' @return A character vector with the letters rearranged.
#' @seealso \code{\link{cld2}()}.
#' @examples
#'
#' # Toy data.
#' set.seed(4321)
#' td <- data.frame(trt = rep(sample(1:8), each = 5))
#' td$y <- rnorm(nrow(td), mean = sort(td$trt), sd = 2)
#'
#' plot(y ~ trt, data = td)
#'
#' # Fit the model.
#' td$trt <- factor(td$trt)
#' m0 <- lm(y ~ trt, data = td)
#' anova(m0)
#' summary(m0)
#'
#' library(multcomp)
#' library(doBy)
#'
#' X <- LSmatrix(m0, effect = "trt")
#' rownames(X) <- levels(td$trt)
#' Xc <- apc(X)
#'
#' g <- summary(glht(m0, linfct = Xc),
#'              test = adjusted(type = "fdr"))
#'
#' res <- data.frame(trt = levels(td$trt),
#'                   mean = X %*% coef(m0))
#'
#' let <- cld2(g)
#' res$cld2 <- let$mcletters$Letters
#' res[order(res$mean, decreasing = TRUE), ]
#'
#' res$let2 <- ordered_cld(res$cld2, res$mean)
#' res[order(res$mean, decreasing = TRUE), ]
#'
#' \dontrun{
#'
#' library(latticeExtra)
#' library(grid)
#'
#' ci <- as.data.frame(
#'     confint(glht(m0, linfct = X),
#'             calpha = univariate_calpha())$confint)
#' ci <- cbind(res, ci)
#'
#' segplot(reorder(trt, Estimate) ~ lwr + upr,
#'         centers = Estimate,
#'         data = ci,
#'         draw = FALSE,
#'         cld = ci$let2,
#'         par.settings = list(layout.widths = list(right.padding = 7))) +
#'     layer(panel.text(x = centers,
#'                      y = z,
#'                      labels = sprintf("%0.2f %s",
#'                                       centers,
#'                                       cld),
#'                      pos = 3))
#'
#' ocld <- with(ci[order(ci$Estimate), ],
#'      ordered_cld(cld2, Estimate))
#' x <- attr(ocld, "ind")
#' index <- which(x, arr.ind = TRUE)
#' trellis.focus("panel", column = 1, row = 1, clip.off = TRUE)
#' xcor <- 1.03 + (index[, 2] - 1)/50
#' grid.segments(x0 = unit(xcor, "npc"),
#'               x1 = unit(xcor, "npc"),
#'               y0 = unit(index[, 1] + 0.5, units = "native"),
#'               y1 = unit(index[, 1] - 0.5, units = "native"),
#'               gp = gpar(lwd = 2, col = "blue"))
#' trellis.unfocus()
#'
#' }
#'
ordered_cld <- function(let, means = let) {
    or <- order(means, decreasing = TRUE)
    let <- as.character(let[or])
    s <- strsplit(let, "")
    u <- unlist(s)
    ul <- unique(u)
    UL <- LETTERS[seq_along(ul)]
    l <- sapply(s, FUN = function(i) {
        paste(sort(UL[match(i, table = ul)]), collapse = "")
    })
    x <- tolower(l[order(or)])
    UL <- tolower(UL)
    attr(x, "match") <- cbind("before" = ul,
                              "after" = UL)
    attr(x, "ind") <- sapply(UL, FUN = grepl, x = x)
    return(x)
}

#' @name radial_cld
#' @author Walmes Zeviani, \email{walmes@@ufpr.r}.
#' @export
#' @title Radial Plot for a Compact Letter Display Vector
#' @description This function does a radial plot based on the vector of
#'     letters resulted from pairwise comparisons.
#' @param cld Character vector with strings of letters that indicates
#'     which pair of treatment cells are not different.
#' @param labels Vector of text to be annotated next each point.
#' @param col Vector of colors to be used in the segments that joint
#'     points.
#' @param means Numeric vector with the estimated means of treatment
#'     cells. It is used to place points at distances proportional to
#'     the differences on means.
#' @param perim Logical value (default is \code{FALSE}) that indicates
#'     weather draw or not a circle in the perimeter passing by the
#'     points.
#' @param legend Logical value (default is \code{TRUE}) that indicates
#'     weather daraw or not the legend.
#' @return None is returned, only the plot is done.
#' @seealso \code{\link{cld2}()}.
#' @importFrom utils combn
#' @examples
#'
#' set.seed(4321)
#' td <- data.frame(trt = rep(sample(1:20), each = 5))
#' td$y <- rnorm(nrow(td), mean = 0.15 * sort(td$trt), sd = 1)
#'
#' plot(y ~ trt, data = td)
#'
#' # Fit the model.
#' td$trt <- factor(td$trt)
#' m0 <- lm(y ~ trt, data = td)
#' anova(m0)
#' summary(m0)
#'
#' library(multcomp)
#' library(doBy)
#'
#' X <- LSmatrix(m0, effect = "trt")
#' rownames(X) <- levels(td$trt)
#'
#' ci <- apmc(X, m0, focus = "trt", test = "fdr")
#' ci$cld <- with(ci, ordered_cld(cld, fit))
#' ci <- ci[order(ci$fit, decreasing = TRUE), ]
#'
#' library(latticeExtra)
#'
#' segplot(reorder(trt, fit) ~ lwr + upr,
#'         centers = fit,
#'         data = ci,
#'         draw = FALSE,
#'         cld = ci$cld) +
#'     layer(panel.text(x = centers,
#'                      y = z,
#'                      labels = sprintf("%0.2f %s",
#'                                       centers,
#'                                       cld),
#'                      pos = 3))
#'
#' radial_cld(cld = ci$cld)
#' radial_cld(cld = ci$cld, means = ci$fit, perim = TRUE)
#' radial_cld(cld = ci$cld, col = 1:3)
#' radial_cld(cld = ci$cld, col = 1:3)
#' radial_cld(cld = ci$cld, labels = sprintf("%0.2f %s", ci$fit, ci$cld))
#'
radial_cld <- function(cld,
                       labels = cld,
                       col = NULL,
                       means = NULL,
                       perim = FALSE,
                       legend = TRUE) {
    if (is.null(means)) {
        s <- seq(from = 0,
                 to = 2 * pi,
                 length.out = length(cld) + 1)[-1]
    } else {
        ext <- (2 * pi)/c(length(cld), 1)
        m <- means - min(means)
        m <- m/max(m)
        s <- ext[1] + diff(ext) * m
        s <- rev(s)
    }
    sincos <- cbind(sin = sin(s), cos = cos(s))
    # Quais as letras únicas formadoras das strings?
    u <- unique(unlist(strsplit(cld, split = "")))
    if (is.null(col)) {
        col <- palette()
    }
    if (length(col) != length(u)) {
        warning(paste("Length of vector `col` is different",
                      "of the number of unique letters.",
                      "Colors will be recycled."))
    }
    col <- col[seq_along(u) %% length(col) + 1]
    # Membros da mesma família compartilham a mesma letra.
    fam <- sapply(u, grepl, x = cld)
    plot(x = NULL,
         y = NULL,
         xlim = 1.2 * c(-1, 1),
         ylim = 1.2 * c(-1, 1),
         asp = 1,
         axes = FALSE,
         ann = FALSE)
    if (perim) {
        circ <- seq(0, 2 * pi, length.out = 60)
        lines(x = sin(circ), y = cos(circ), col = "gray", lty = 3)
    }
    for (i in 1:ncol(fam)) {
        cb <- combn(x = which(fam[, i]), m = 2)
        apply(cb,
              MARGIN = 2,
              FUN = function(index) {
                  segments(x0 = sincos[index[1], 1],
                           x1 = sincos[index[2], 1],
                           y0 = sincos[index[1], 2],
                           y1 = sincos[index[2], 2],
                           col = col[i],
                           lwd = 2,
                           lty = 2)
              })
    }
    points(x = sincos[, 1], y = sincos[, 2])
    if (legend) {
        legend("topright",
               legend = u,
               col = col,
               lty = 2,
               lwd = 2,
               bty = "n")
    }
    text(sincos[, 1],
         sincos[, 2],
         labels = labels,
         pos = ifelse(sincos[, 1] > 0, 4, 2))
}


#' @title Delta Method
#' @author Wagner Hugo Bonat, \email{wbonat@@ufpr.br}
#'
#' @description Compute standard errors for functions of model parameters
#' using the delta method.
#'
#' @param g A function (string like formula) of model parameters. 
#' @param mean Vector of parameter estimates.
#' @param cov Variance-covariance matrix.
#' @param ses Logical. If TRUE returns the standard error, otherwise
#' return the new variance-covariance.
#' @keywords internal
#' @details It is an internal function useful in general for summary 
#' function associated with Twin models.

deltamethod <- function (g, mean, cov, ses = TRUE) {
    cov <- as.matrix(cov)
    n <- length(mean)
    if (!is.list(g)) 
        g <- list(g)
    if ((dim(cov)[1] != n) || (dim(cov)[2] != n)) 
        stop(paste("Covariances should be a ", n, " by ", n, 
                   " matrix"))
    syms <- paste("x", 1:n, sep = "")
    for (i in 1:n) assign(syms[i], mean[i])
    gdashmu <- t(sapply(g, function(form) {
        as.numeric(attr(eval(deriv(form, syms)), "gradient"))
    }))
    new.covar <- gdashmu %*% cov %*% t(gdashmu)
    if (ses) {
        new.se <- sqrt(diag(new.covar))
        new.se
    }
    else new.covar
}


# logLik function for object of mlm class ----------------------------

logLik.mlm <- function(object,...)
{
  resids <- residuals(object)
  n <- nrow(resids)
  Sigma_ML <- crossprod(resids) /n
  ans <- sum(dmvnorm(resids, sigma=Sigma_ML, log=T))
  
  df <- length(coef(object)) + nrow(Sigma_ML) * (nrow(Sigma_ML) + 1) / 2
  attr(ans, "nobs") <- n
  attr(ans, "df") <- df
  class(ans) <- "logLik"
  ans
}

# END ----------------------------------------------------------------

