!=======================================================================
subroutine cubicBsplines(x, nx, xl, xr, ndx, B)
!=======================================================================
! Input: x(nx): values where to compute the cubic B-splines basis ;
!        xl, xr, ndx: typically knots = seq(xl, xr, by=(xr-xl)/ndx)
!
! Output: B(nx, ndx+3)
!
! By P. Lambert (2010) inspired by Paul Eilers R code.
!

  implicit none
  integer nx, ndx
  double precision x(nx), xl, xr, B(nx,ndx+3)
  
  integer nknots, i, j
  double precision knots(ndx+7), dx, temp, cub

  nknots = ndx + 7
  dx = (xr - xl) / ndx

  knots(1) = xl - 3.*dx
  do i= 2, nknots
     knots(i) = knots(i-1) + dx
  end do

  do i = 1, nx
     do j = 1, nknots-4
        temp = 0.
        cub = x(i) - knots(j)
        if (cub .gt. 0.) then
           temp = temp + cub*cub*cub
           cub = x(i) - knots(j+1)
           if (cub .gt. 0.) then
              temp = temp - 4.*cub*cub*cub
              cub = x(i) - knots(j+2)
              if (cub .gt. 0.) then
                 temp = temp + 6.*cub*cub*cub
                 cub = x(i) - knots(j+3)
                 if (cub .gt. 0.) then
                    temp = temp - 4.*cub*cub*cub
                    cub = x(i) - knots(j+4)
                    if (cub .gt. 0.) then
                       temp = temp + cub*cub*cub
                    end if
                 end if
              end if
           end if
        end if
        B(i,j) = temp / (6.*dx*dx*dx)
        if (abs(B(i,j)) .lt. 1e-10) B(i,j)= 0.
     end do
  end do
end subroutine cubicBsplines

!=======================================================================
subroutine MHstep(np, pcur, lcur, pprop, lprop, m, burnin, sigma0, sigmacur, &
    targetRate, accept)
!=======================================================================
!
! Goal : indicates whether one accepts the proposed M-H step
!        and updates the std deviation of the normal shocks if
!        the iteration number m > 5 and m < burnin.
!
! Input:
!        pcur: current value of p of length np
!        lcur: current value of logpost
!        pprop: proposed value for p of length np
!        lprop: proposed value for logpost
!        m : iteration number
!        brunin : length of burnin
!        sigma0 : std deviation of normal shocks at iteration 1
!        sigmacur : std deviation of normal shocks at current iteration
!        targetRate : targetted acceptance rate in M-H algorithm
!
! Output:
!        pcur : becomes pprop if acceptance, unchanged otherwise
!        lcur : becomes lprop if acceptance, unchanged otherwise
!        sigmacur: new std deviation after this M-H step
!        accept : 0-1 indicator of acceptance of M-H step
!
! (P. Lambert, 2009)
!
 implicit none
 integer np, m, burnin, accept
 double precision pcur(np), lcur, pprop(np), lprop, sigma0, sigmacur, &
      targetRate

 double precision lalpha, alpha, rnd, sigmaprop

 lalpha = lprop - lcur
 if (lalpha .gt. 0.d0) lalpha = 0.d0
 if (lalpha .lt. -10.d0) then
    alpha = 0.d0
 else
    alpha = exp(lalpha)
 end if

 call random_number(rnd)
 if (rnd .le. alpha) then
   pcur(:) = pprop(:)
   lcur = lprop
   accept = 1
 else
   accept = 0
 endif

 if ((m > 5) .and. (m < burnin)) then
    sigmaprop = sigmacur + sigma0 * (alpha - targetRate)/sqrt(dble(m))
    if (sigmaprop .ne. sigmaprop) sigmaprop = sigmacur ! Check if is NaN...
    if (sigmaprop .lt. 0) sigmaprop = .9d0*sigmacur
    sigmacur = sigmaprop
 endif
end subroutine MHstep

subroutine inverse(res, n, A, info)
  integer n, lda, lwork
  double precision A(n,n), res(n,n), work(n)
  integer ipiv(n), info
  lda = n
  lwork = n
  res(1:n, 1:n) = A(1:n,1:n)
  call dgetrf (n, n, res, lda, ipiv, info ) ! Compute the LU factorization ...
  call dgetri(n, res, lda, ipiv, work, lwork, info) ! ... to get the inverse of the matrix.
end subroutine inverse


! Get the coefficient of (a + b)^order
subroutine coefbin(order, res)
	
   implicit none
   integer :: i,j, order
   double precision :: res(order + 1)
   double precision :: temp(order, order+1)
	
   temp(1,1) = 1
   temp(1,2) = 1

   do i = 2, order
      temp(i, 1) = 1
      do j = 2,i
         temp(i,j) = temp(i-1, j-1) + temp(i-1, j)
      end do 
      temp(i, i+1) = 1
   end do 		
   res = temp(order, :)      
end subroutine coefbin

! Penalty matrix 
subroutine penmat(nknots, degree, order, res)
   implicit none
   integer :: i, nknots, degree, order
   double precision :: coef(order + 1), tempM(nknots+degree-order, nknots + degree)
   double precision  :: tempV(nknots + degree), res(nknots+degree, nknots + degree)
	
   do i = 1,(nknots+degree)
      tempV(i) = 0
   end do 
	
   call coefbin(order, coef) 
   do i = 1,order+1
      coef(i) = coef(i) * (-1)**(i+1)
   end do 
	
   do i = 1,(nknots+degree-order)
      tempM(i,:) = tempV
      tempM(i, i:(order+1)) = coef
   end do 
   res = matmul(transpose(tempM), tempM)	 
end subroutine penmat

subroutine rnorm(n, x)
  integer n, i
  double precision normrnd, x(n)
  call rndstart()
  do i = 1, n
     x(i) = normrnd()
  end do
  call rndend()
end subroutine rnorm

subroutine rgamma(n, ans, a, b)
  integer n, i
  double precision ans(n), a, b, bb
  bb = 1/b
  call rndstart()
  do i = 1, n
     call gammarnd(ans(i), a, bb)
  end do
  call rndend()
end subroutine rgamma

subroutine cholesky(res, n, A)
  integer n, lda, info, i
  double precision A(n,n), res(n,n)
  character*1 uplo 
  uplo = 'L'
  lda = n
  res(1:n,1:n) = A(1:n,1:n)
  call DPOTRF(UPLO, N, res, LDA, INFO) ! Cholesky using LAPACK
! See   http://www.netlib.org/lapack/double/dpotrf.f   for documentation
  do i = 1, n-1
     res(i,(i+1):n) = 0.
  end do
end subroutine cholesky

subroutine FPTRC(&
& n_splines_BD, n_splines_estimate_BD, rank_penalty, n_splines_Cov_Cure, n_splines_Cov_Cox , n_splines_cov_one,&
& n_cov_cont_Cure, n_cov_cont_Cox, Bobs, Bmiddle, BcovCure, BcovCox, & 
& penalty_BD, penalty_Cov, & 
& sigma_Cure, sigma_Cox, & 
& n, eventInd, &  
& Npart, delta, upto, &
& X, W, nalpha, nbeta, &
& nu, a_delta, b_delta, sd_Cov, &
& initphi_BD, initpen_BD, initdelta_BD, &
& initalpha, &
& initphi_Cure, initpen_Cure, initdelta_Cure, &  
& initbeta, &
& initphi_Cox, initPen_Cox, initdelta_Cox, &
& InitSDprop_Cure, InitSDprop_Cox, InitSDprop_reg, &
& iteration, updatesd, &  
& phi_BD, pen_BD, delta_BD, & 
& alpha, &
& phi_Cure, pen_Cure, delta_Cure, &
& beta, & 
& phi_Cox, pen_Cox, delta_Cox, &
& accept)

   implicit none 
   ! input 
   integer :: n_splines_BD, n_splines_estimate_BD, n_splines_Cov_Cure, n_splines_Cov_Cox, n, eventInd(n), Npart
   integer :: iteration, updatesd, n_splines_cov_one, rank_penalty, upto(n), nalpha, nbeta
   integer :: n_cov_cont_Cure, n_cov_Cont_Cox

   double precision :: Bobs(n, n_splines_BD), Bmiddle(Npart, n_splines_BD), BcovCure(n, n_splines_Cov_Cure)
   double precision :: BcovCox(n, n_splines_Cov_Cox)
   double precision :: penalty_BD(rank_penalty,rank_penalty), penalty_Cov(n_splines_cov_one,n_splines_cov_one)
   double precision :: sigma_Cure(n_splines_Cov_Cure + 1, n_splines_Cov_Cure + 1)
   double precision :: sigma_Cox(n_splines_estimate_BD + n_splines_Cov_Cox, n_splines_estimate_BD + n_splines_Cov_Cox)

   double precision :: delta
   double precision :: X(n, nalpha), W(n, nbeta)

   double precision :: nu, a_delta, b_delta, sd_Cov

   double precision :: initphi_BD(n_splines_BD), initpen_BD, initdelta_BD

   double precision :: initalpha(nalpha)

   double precision :: initphi_Cure(n_splines_Cov_Cure),  initpen_Cure(n_cov_cont_Cure), initdelta_Cure(n_cov_cont_Cure)

   double precision :: initbeta(nbeta)

   double precision :: initphi_Cox(n_splines_Cov_Cox), initpen_Cox(n_cov_cont_Cox), initdelta_Cox(n_cov_cont_Cox)

   double precision :: InitSDprop_Cure, InitSDprop_Cox, InitSDprop_reg 
    
   double precision :: phi_BD(n_splines_BD,iteration), pen_BD(iteration+1), delta_BD(iteration+1) 

   double precision :: alpha(nalpha, iteration)

   double precision :: phi_Cure(n_splines_Cov_Cure, iteration), pen_Cure(n_cov_cont_Cure, iteration+1)
   double precision :: delta_Cure(n_cov_cont_Cure, iteration+1)

   double precision :: beta(nbeta, iteration)
   
   double precision :: phi_Cox(n_splines_Cov_Cox, iteration), pen_Cox(n_cov_cont_Cox, iteration+1)
   double precision :: delta_Cox(n_cov_cont_Cox, iteration+1)

   integer :: accept(3, iteration) 


! End of the declaration of the input of the subroutine. 
!_________________________________

   ! Count variables for the different loops. 
   integer :: cpt, NB, NL

   ! Some integer variables needed during the subroutine.
   integer :: one
       
   double precision :: phiProp_BD(n_splines_BD), phiProp_Cure(n_splines_Cov_Cure), phiProp_Cox(n_splines_Cov_Cox) 
   double precision :: alphaProp(nalpha), betaprop(nbeta) 

   double precision :: phiCur_BD(n_splines_BD) , phiCur_Cure(n_splines_Cov_Cure), phiCur_Cox(n_splines_Cov_Cox) 
   double precision :: alphaCur(nalpha), betaCur(nbeta)

   double precision :: CurSDprop_Cox, CurSDprop_Cure, CurSDprop_reg 

   ! cholesky decomposition of the naive correlation struture of the spline parameters  
   double precision :: L_Cure(1+n_splines_Cov_Cure, 1+n_splines_Cov_Cure)
   double precision :: L_Cox(n_splines_estimate_BD + n_splines_Cov_Cox, n_splines_estimate_BD + n_splines_Cov_Cox)

   ! log Posterior function and current and proposal value of the log_post
   double precision :: logpost, lcur, lprop
     
   double precision :: updateA_delta_BD, updateB_delta_BD, updateApen_BD, updateBpen_BD
   double precision :: updateA_delta_Cure, updateB_delta_Cure, updateApen_Cure, updateBpen_Cure
   double precision :: updateA_delta_Cox, updateB_delta_Cox, updateApen_Cox, updateBpen_Cox

   double precision :: Cox_prop(n_splines_estimate_BD + n_splines_Cov_Cox) 
   double precision :: Cox_Cur(n_splines_estimate_BD + n_splines_Cov_Cox)
   double precision :: prop_Cox(n_splines_estimate_BD + n_splines_Cov_Cox)

   double precision :: Cure_prop(n_splines_Cov_Cure + 1) 
   double precision :: Cure_Cur(n_splines_Cov_Cure + 1)
   double precision :: prop_Cure(n_splines_Cov_Cure + 1)

   double precision :: prop_reg(nalpha-1+nbeta), reg_prop(nalpha-1+nbeta), reg_Cur(nalpha-1+nbeta) 

   double precision :: targetRate, targetRateMult 
   integer :: i 
   
   targetRate = 0.44
   targetRateMult = 0.23

   ! Tools numbers
   one = 1  

   ! Cholesky decomposition of the naive correlation structure of the spline parameters. 
   call cholesky(L_Cure, n_splines_Cov_Cure + 1, sigma_Cure)   
   call cholesky(L_Cox, n_splines_estimate_BD + n_splines_Cov_Cox, sigma_Cox)   
 
   ! Default starting sd for the proposal. 
   CurSDprop_Cox = InitSDprop_Cox 
   CurSDprop_Cure = InitSDprop_Cure
   CurSDprop_reg = InitSDprop_reg

   phiCur_BD = initphi_BD 
   pen_BD(1) = initpen_BD
   delta_BD(1) = initdelta_BD

   phiCur_Cure = initphi_Cure
   pen_Cure(1:n_cov_cont_Cure,1) = initpen_Cure(1:n_cov_cont_Cure)
   delta_Cure(1:n_cov_cont_Cure,1) = initdelta_Cure(1:n_cov_cont_Cure)

   phiCur_Cox = initphi_Cox
   pen_Cox(1:n_cov_cont_Cox,1) = initpen_Cox(1:n_cov_cont_Cox)
   delta_Cox(1:n_cov_cont_Cox,1) = initdelta_Cox(1:n_cov_cont_Cox)

   alphaCur = initalpha
   betaCur = initbeta
 
   ! Initial value of the likelihood
   lcur = logPost(n, eventInd, n_splines_BD, rank_penalty,  phiCur_BD, Bobs, Bmiddle, Npart, delta, upto, penalty_BD, & 
        & pen_BD(1), x, alphaCur, nalpha, W, betaCur ,nbeta, sd_Cov,  & 
     & n_splines_Cov_Cure, n_splines_Cov_Cox, n_splines_cov_one, n_cov_cont_Cure, n_cov_Cont_Cox, phiCur_Cure, phiCur_cox, &  
         & BCovCure, BcovCox, penalty_Cov, pen_Cure(:,1), pen_Cox(:,1))

   write(*,*) iteration, "iterations remaining"
   write(*,*) " " 

   ! Iteration for each step (update of the sd proposal and the final posterior chains)
   do cpt = 1, iteration

      if( modulo(cpt,10000) == 0 ) then 

         write(*,*) (iteration-cpt), "iterations remaining" 
         write(*,*) " "

      end if 

      Cox_Cur( 1:n_splines_estimate_BD ) = phiCur_BD( 1:n_splines_estimate_BD )
      Cox_Cur( (n_splines_estimate_BD + 1) : (n_splines_estimate_BD + n_splines_Cov_cox) ) = phiCur_Cox

      Cure_Cur( 1:n_splines_cov_Cure ) = phiCur_Cure 
      Cure_Cur( n_splines_cov_Cure + 1 ) = alphaCur(1) 

      reg_Cur( 1:(nalpha-1) ) = alphaCur(2:nalpha) 
      reg_Cur( nalpha:(nalpha+nbeta-1) ) = betaCur 

      ! Metropolis step for the spline in the BD.
      call rnorm(n_splines_estimate_BD + n_splines_Cov_Cox, prop_Cox)

      Cox_Prop = Cox_Cur + CurSDprop_Cox * matmul(L_Cox, prop_Cox)

      phiProp_BD(1:n_splines_estimate_BD) = Cox_prop(1:n_splines_estimate_BD) 
      phiProp_BD( (n_splines_estimate_BD+1) : n_splines_BD ) = 10.d0

      do i = 1,n_cov_cont_Cox

  phiProp_Cox( ( (i-1) * n_splines_Cov_One + 1 ) :  (i* n_splines_Cov_One) ) &
  & = Cox_prop( ( (i-1) * n_splines_Cov_One + 1 + n_splines_estimate_BD ) :  ( i * n_splines_Cov_One + n_splines_estimate_BD ) ) & 
  & - sum( & 
  & Cox_prop( ( (i-1) * n_splines_Cov_One + 1 + n_splines_estimate_BD ) :  ( i * n_splines_Cov_One + n_splines_estimate_BD ) ) & 
  & / n_splines_Cov_one )

      end do 
            
      lprop = logPost(n, eventInd, n_splines_BD, rank_penalty, phiProp_BD, Bobs, Bmiddle, Npart, delta, upto, penalty_BD,&
              & pen_BD(cpt), x, alphaCur, nalpha, W, betaCur ,nbeta,sd_Cov, & 
& n_splines_Cov_Cure, n_splines_Cov_Cox, n_splines_cov_one, n_cov_cont_Cure, n_cov_cont_Cox, phiCur_Cure, phiProp_Cox, & 
              & BcovCure, BcovCox, penalty_Cov, pen_Cure(:,cpt), pen_Cox(:,cpt))

      call MHstep(n_splines_estimate_BD + n_splines_Cov_cox, Cox_cur, lcur, Cox_Prop, lprop, cpt, updatesd, & 
                   & InitSDprop_Cox, CurSDprop_Cox, targetRateMult, accept(1, cpt))

      phiCur_BD(1:n_splines_estimate_BD) = Cox_cur(1:n_splines_estimate_BD) 

      do i = 1, n_cov_cont_Cox 

  phiCur_Cox( ( (i-1) * n_splines_Cov_One + 1 ) :  (i* n_splines_Cov_One) ) & 
  & = Cox_cur( ( (i-1) * n_splines_Cov_One + 1 + n_splines_estimate_BD ) :  ( i * n_splines_Cov_One + n_splines_estimate_BD ) ) &
  & - sum( &
  & Cox_cur( ( (i-1) * n_splines_Cov_One + 1 + n_splines_estimate_BD ) :  ( i * n_splines_Cov_One + n_splines_estimate_BD ) ) & 
  & / n_splines_Cov_one) 
 
      end do 

      phi_Cox(:, cpt) = phiCur_Cox
      phi_BD(:, cpt) = phiCur_BD

      ! Metropolis step for the regressors influencing the cure probability. 
      call rnorm(n_splines_Cov_Cure + 1, prop_Cure)

      Cure_Prop = Cure_Cur + CurSDprop_Cure * matmul(L_Cure, prop_Cure)

      do i = 1, n_cov_cont_Cure 

         phiProp_Cure( ( (i-1) * n_splines_Cov_One + 1 ) :  (i* n_splines_Cov_One) ) & 
         & = Cure_Prop( ( (i-1) * n_splines_Cov_One + 1 ) :  (i* n_splines_Cov_One) ) &
         &  - sum( Cure_Prop( ( (i-1) * n_splines_Cov_One + 1 ) :  (i* n_splines_Cov_One) ) & 
         & / n_splines_cov_one )
     
      end do 

      alphaProp(1) = Cure_prop(n_splines_cov_Cure + 1)
      alphaProp(2:nalpha) = alphaCur(2:nalpha)   

      lprop = logPost(n, eventInd, n_splines_BD, rank_penalty, phiCur_BD, Bobs, Bmiddle, Npart, delta, upto, penalty_BD,&
              & pen_BD(cpt), x, alphaProp, nalpha, W, betaCur ,nbeta,sd_Cov, & 
& n_splines_Cov_Cure, n_splines_Cov_Cox, n_splines_cov_one, n_cov_cont_Cure, n_cov_cont_Cox, phiProp_Cure, phiCur_Cox, & 
              & BcovCure, BcovCox, penalty_Cov, pen_Cure(:,cpt),  pen_Cox(:,cpt))

      call MHstep(n_splines_Cov_Cure + 1, Cure_cur, lcur, cure_Prop, lprop, cpt, updatesd, InitSDprop_Cure, &
               &  CurSDprop_Cure, targetRateMult, accept(2, cpt))

      do i = 1, n_cov_cont_Cure 

         phiCur_cure( ( (i-1) * n_splines_Cov_One + 1 ) :  (i* n_splines_Cov_One) ) &
         & = Cure_Cur( ( (i-1) * n_splines_Cov_One + 1 ) :  (i* n_splines_Cov_One) ) & 
         & - sum( Cure_Cur( ( (i-1) * n_splines_Cov_One + 1 ) :  (i* n_splines_Cov_One) ) & 
         & / n_splines_cov_one ) 

      end do 

      alphaCur(1) = Cure_Cur(n_splines_Cov_Cure + 1 )
      alphaProp(1) = Cure_Cur(n_splines_Cov_Cure + 1 )

      phi_Cure(:, cpt) = phiCur_Cure     

      call rnorm(nalpha-1+nbeta, prop_reg)

      reg_Prop = reg_Cur + CurSDprop_reg * prop_reg

      alphaProp( 2:nalpha ) = reg_prop( 1:(nalpha-1) ) 
      betaProp = reg_prop( nalpha:(nalpha+nbeta-1) ) 

      lprop = logPost(n, eventInd, n_splines_BD, rank_penalty, phiCur_BD, Bobs, Bmiddle, Npart, delta, upto, penalty_BD,&
               & pen_BD(cpt), x, alphaProp, nalpha, W, betaProp ,nbeta,sd_Cov, & 
& n_splines_Cov_Cure, n_splines_Cov_Cox, n_splines_cov_one, n_cov_cont_Cure, n_cov_cont_Cox, phiCur_Cure, phiCur_Cox, & 
             & BcovCure, BcovCox, penalty_Cov, pen_Cure(:,cpt), pen_Cox(:,cpt))

     call MHstep(nalpha-1+nbeta, reg_cur, lcur, reg_Prop, lprop, cpt, updatesd, InitSDprop_reg, &
               &  CurSDprop_reg, targetRateMult, accept(3, cpt))

      alphaCur( 2:nalpha ) = reg_cur( 1:(nalpha-1) ) 
      betaCur = reg_cur( nalpha:(nalpha+nbeta-1) ) 

      alpha(:, cpt) = alphaCur
      beta(:, cpt) = betaCur

      ! Gibbs step for the penalty parameters of the splines in the BD. 
       updateApen_BD = (nu + dble(rank_penalty))/2.d0
       updateBpen_BD = (nu + delta_BD(cpt) + DOT_PRODUCT( phiCur_BD(1:rank_penalty), & 
              &  matmul( penalty_BD, phiCur_BD(1:rank_penalty) ) ) ) / 2.d0
       call rgamma(one, pen_BD(cpt+1), updateApen_BD, updateBpen_BD)

       updateA_delta_BD = a_delta  + nu/2.d0
       updateB_delta_BD = b_delta  + nu*pen_BD(cpt+1)/2.d0
       call rgamma(one, delta_BD(cpt+1), updateA_delta_BD, updateB_delta_BD)

      do i = 1, n_cov_cont_Cox 

         updateApen_Cox = (nu + dble(n_splines_Cov_one))/2.d0

         updateBpen_Cox = (nu + delta_Cox(i, cpt) + DOT_PRODUCT(phiCur_Cox( (1+(i-1)*n_splines_cov_one) : (i*n_splines_cov_one) ), &
         &  matmul( penalty_Cov, phiCur_Cox( (1+(i-1)*n_splines_cov_one) : (i*n_splines_cov_one) ) ) ) ) / 2.d0

         call rgamma(one, pen_Cox(i, cpt+1), updateApen_Cox, updateBpen_Cox)

       updateA_delta_Cox = a_delta  + nu/2.d0
       updateB_delta_Cox = b_delta  + nu*pen_Cox(i, cpt+1)/2.d0
       call rgamma(one, delta_Cox(i, cpt+1), updateA_delta_Cox, updateB_delta_Cox)

      end do

      do i = 1, n_cov_cont_Cure 

      updateApen_Cure = (nu + dble(n_splines_Cov_one))/2.d0
      updateBpen_Cure = (nu + delta_Cure(i, cpt) + DOT_PRODUCT(phiCur_Cure( (1+(i-1)*n_splines_cov_one) : (i*n_splines_cov_one) ), &
         &  matmul( penalty_Cov, phiCur_Cure( (1+(i-1)*n_splines_cov_one) : (i*n_splines_cov_one) ) ) ) ) / 2.d0

      call rgamma(one, pen_Cure(i, cpt+1), updateApen_Cure, updateBpen_Cure)

      updateA_delta_Cure = a_delta  + nu/2.d0
      updateB_delta_Cure = b_delta  + nu*pen_Cure(i, cpt+1)/2.d0
      call rgamma(one, delta_Cure(i, cpt+1), updateA_delta_Cure, updateB_delta_Cure)

   end do 

       ! New value of the likelihood with the new value of the penalty parameters of the splines in the BD. 
       lcur = logPost(n, eventInd, n_splines_BD, rank_penalty, phiCur_BD, Bobs, Bmiddle, Npart, delta, upto, penalty_BD,& 
	  & pen_BD(cpt+1), x, alphaCur, nalpha, W, betaCur ,nbeta, sd_Cov,& 
& n_splines_Cov_Cure, n_splines_Cov_Cox, n_splines_cov_one, n_cov_cont_Cure, n_cov_cont_Cox, phiCur_Cure, phiCur_Cox, & 
                & BcovCure, BcovCox, penalty_Cov, pen_Cure(:,cpt+1), pen_Cox(:,cpt+1))            
      end do

end subroutine FPTRC


double precision function logPost(& 
& n, eventInd, & ! information available in the data set
& n_splines_BD, rank_penalty, phi_BD, Bobs, Bmiddle, Npart, delta, upto, & ! tools for baseline distribution
& penalty_BD, pen_BD, & ! penalty
& X, alpha, nalpha, & ! Cure parameters
& W, beta,nbeta, sd_Cov,& ! Parameters in Cox model for the latent distribution
& n_splines_Cov_Cure, n_splines_Cov_Cox ,n_splines_cov_one, n_cov_cont_Cure, n_Cov_Cont_Cox, phi_Cure, phi_Cox, &
& BCovCure, BcovCox, penalty_Cov, pen_Cure, pen_Cox)

   implicit none
   integer :: n, eventInd(n), n_splines_cov_one, n_cov_cont_Cure, n_cov_cont_Cox, rank_penalty !cf. line 1 declaration function 
 
   integer :: n_splines_BD, Npart, upto(n) ! cf. line 2 declaration function
   double precision :: phi_BD(n_splines_BD), Bobs(n, n_splines_BD), Bmiddle(Npart, n_splines_BD), delta  

   double precision :: penalty_BD(rank_penalty, rank_penalty), pen_BD(1) ! cf. line 3 declaration function
   
   integer :: nalpha ! cf. line 4 declaration function
   double precision :: X(n, nalpha), alpha(nalpha) ! cf. line 4 declaration function

   integer :: nbeta ! cf. line 5 declaration function
   double precision :: W(n, nbeta), beta(nbeta) ! cf. line 5 declaration function

   integer :: n_splines_Cov_Cure, n_splines_Cov_Cox
   double precision :: phi_Cure(n_splines_Cov_Cure), phi_Cox(n_splines_Cov_Cox)
   double precision :: BCovCure(n, n_splines_Cov_Cure), BCovCox(n, n_splines_Cov_Cox)

   double precision :: penalty_Cov(n_splines_cov_one,  n_splines_cov_one), pen_Cure(n_cov_cont_Cure,1), pen_Cox(n_cov_cont_Cox,1)
   double precision :: sd_Cov

 
  ! count variable for the loop
   integer ::  i

   ! tools for the cumulative hazard
   double precision ::  temp(Npart), tempcum(Npart)

   ! Baseline distribution, several functions 
   double precision :: h(n), Hcum(n), s(n), F(n), density(n)

   ! Mean parameters of the number of carcogenic cells
   double precision :: theta(n)

   ! Tools variable for the calculation of the log_post. 
   double precision :: t1, t2, t3, t4, t5, t6, t7

   ! LOG-LIKELIHOOD
   h = exp(matmul(Bobs, phi_BD))*exp(matmul(W, beta)+matmul(BCovCox, phi_Cox))

   temp = exp(matmul(Bmiddle, phi_BD))*delta
   tempcum(1) = temp(1)

   do i = 2,Npart
   tempcum(i) = tempcum(i-1) + temp(i) 
   end do 

   Hcum = tempcum(upto) 
   Hcum = Hcum*exp(matmul(W, beta)+matmul(BCovCox, phi_Cox))
   S = exp(-Hcum)

   F = 1.0 - S
   density = h * s

   do i = 1, n 

      if (density(i) < 0.0000001) then 
          density(i) = 0.0000001 
      end if 

   end do 

   theta = exp( matmul(x, alpha) + matmul(BcovCure, phi_Cure) )

   t1 = dot_product(-theta, F)
   t2 = dot_product(dble(eventInd), log(theta))
   t3 = dot_product(dble(eventInd), log(density))
   ! END LOG-LIKELIHOOD
    
   !Spline_BD prior distributionF
   t4 = -pen_BD(1) * DOT_PRODUCT(phi_BD(1:rank_penalty), matmul(penalty_BD, phi_BD(1:rank_penalty))) /2.d0
   !Spline_Cure prior distribution

   t5 = 0.d0

   do i = 1, n_cov_cont_Cure

      t5 = t5 - pen_cure(i,1) * DOT_PRODUCT(phi_Cure( (1+(i-1)*n_splines_cov_one) : (i*n_splines_cov_one) ), & 
           &  matmul(penalty_Cov,phi_Cure( (1+(i-1)*n_splines_cov_one) : (i*n_splines_cov_one) ) ) ) / 2.d0
 
   end do 

   do i = 1, n_cov_cont_Cox


     t5 = t5 -pen_Cox(i,1) * DOT_PRODUCT(phi_Cox( (1+(i-1)*n_splines_cov_one) : (i*n_splines_cov_one) ), &
            & matmul(penalty_Cov,phi_Cox( (1+(i-1)*n_splines_cov_one) : (i*n_splines_cov_one) ) ) ) / 2.d0
   end do

   do i=1, nalpha

      t5 = t5 - (alpha(i)**2 / (2*sd_Cov**2))
   end do 

   do i=1, nbeta
      t5 = t5 - (beta(i)**2 / (2*sd_Cov**2)) 
   end do

   logPost = t1 + t2 + t3 + t4 + t5

end function logPost
