##################################################################################################################
#
# Nonparametric double additive cure survival models: an application to the estimation of the nonlinear 
# effect of age at first parenthood on fertility progression.
# 
# Statistical Modelling - 2018
#
# Code writen by V.Bremhorst (Universite catholique de Louvain - Belgium)
#
# Joint work with : * Michaela Kreyenfeld - Hertie School of Governance Berlin (Germany) 					 			 
#                   * Philippe Lambert - Universite de Liege and Universite catholique de Louvain (Belgium) 
#
# Corresponding author: vincent.bremhorst@uclouvain.be 
#
#
#	************************************************************************************
# 	* 				Main function                                      * 
#	* Compute and return the numerical results for all regression parameters           *
#	* Posterior median as estimator, the posterior std and the 90-95-99% HPD intervals * 
#	* Return also the posterior sample generate using MCMC of all model parameters.    *                                 
#	************************************************************************************
#
####################################################################################################################

SM_BremhorstetAl2017_mainFunction <- function( study.name,

                                               obstime,
                                               Event,

	       	                               Cure,
                                               labels_Cure,
                                               Cox,
					       labels_Cox,

                                               nknots_BD = 17, # Please do not modify
                                               degree_BD = 3, # Please do not modify
                                               order_BD = 3, # Please do not modify

                                               nknots_Cov = 7, # Please do not modify
                                               degree_Cov = 3, # Please do not modify
                                               order_Cov = 2, # Please do not modify

		                               iteration,
                                               burnin
					     ) 


{ 

# Check input
STOP <- FALSE 
cpt_check <- 1 
while(STOP == FALSE & cpt_check <= length(Cure$Categorical) ) {

   if( class(Cure$Categorical[[cpt_check]]) != "factor" ) {
      STOP <- TRUE
      cat( names(Cure$Categorical)[cpt_check], " is not a factor \n")
   }
   cpt_check <- cpt_check + 1
}

cpt_check <- 1 
while(STOP == FALSE & cpt_check <= length(Cox$Categorical) ) {

   if( class(Cox$Categorical[[cpt_check]]) != "factor" ) {
      STOP <- TRUE
      cat( names(Cox$Categorical)[cpt_check], " is not a factor \n")
   }
   cpt_check <- cpt_check + 1
}

if (STOP) { STOP <- TRUE } 

else if ( length(Cure$Categorical) == 0 | length(Cox$Categorical) == 0 ) {

   cat("Error - At least one categorical covariate is needed in each regression part \n") 
}

else if ( length(Cure$Continuous) == 0 | length(Cox$Continuous) == 0 ) {

   cat("Error - At least one continuous covariate needs to be modelled in a flexible way in each regression part \n") 
}

else if ( length(Cure$Continuous) != length(labels_Cure$Continuous)  |
          length(Cox$Continuous) != length(labels_Cox$Continuous)  | 
          length(Cure$Categorical) != length(labels_Cure$Categorical) |
          length(Cox$Categorical) != length(labels_Cox$Categorical)
        ) {

   cat("Error - Labels length not correct \n")
}

else {
# end


   cat("\n\n")

   n <- length(obstime)  # sample size

   # follow-up interval 
   xl <- 0 
   xr <- max(obstime) 
   cat("Observed follow-up interval: [", xl, ";", xr, "]", "\n \n")

   # Number of spline coefficients to be estimated (baseline distribution)
   n_splines_BD <- nknots_BD + degree_BD
   t.event.max <- max(obstime[Event == 1])
   L <- (xr-xl) / (n_splines_BD) 
   n_splines_estimate_BD <- ceiling(t.event.max / L) 
   rank_penalty <- min(n_splines_BD, n_splines_estimate_BD + 2)
   cat("Number of B-spline parameters to be estimated (baseline distribution): ", n_splines_estimate_BD, "\n\n ")

   # Grid for the evaluation of the baseline distribution  
   bins <- 300
   partition <- seq(xl, xr, length = bins+1)
   width <- partition[2] - partition[1]
   middleBins <- partition[1:bins] + (width/2)

   upto <- as.integer(obstime/width) + 1
   upto[which(upto == bins +1)] <- bins

   Bmiddle <- matrix(.Fortran("cubicBsplines", 
                     x = as.double(middleBins), 
                     nx = as.integer(bins), 
                     xl = as.double(xl), 
                     xr = as.double(xr), 
                     ndx = as.integer(nknots_BD), 
                     B = as.double(matrix(0, nrow = bins, ncol = n_splines_BD)))$B, ncol = n_splines_BD)

   Bobs <- matrix(.Fortran("cubicBsplines", 
                  x = as.double(obstime), 
                  nx = as.integer(n), 
                  xl = as.double(xl), 
                  xr = as.double(xr), 
                  ndx = as.integer(nknots_BD), 
                  B = as.double(matrix(0, nrow = n, ncol = n_splines_BD)))$B, ncol = n_splines_BD)

   # Roughness penalty matrix. 
   penalty_BD <- matrix(.Fortran("penmat",
                        nknots = as.integer(rank_penalty - degree_BD), 
                        degree = as.integer(degree_BD),
                        order = as.integer(order_BD), 
                        res = as.double(matrix(0, ncol = rank_penalty, nrow = rank_penalty)))$res, ncol = rank_penalty)

   penalty_BD <- penalty_BD + diag(rep(1, rank_penalty))*10^(-6)


   # Definition of the design matrix X (cf. eq. 2.1) 
   X <- rep(1,n)
   colnamesX <- c("Intercept") 

   for(i in 1:length(Cure$Categorical)) {

      if( class(Cure$Categorical[[i]]) == "factor" ) {

         X <- cbind(X, model.matrix(~Cure$Categorical[[i]])[,-1])
         colnamesX <- c( colnamesX, paste(names(Cure$Categorical)[i], levels(Cure$Categorical[[i]])[-1], sep = "-") )

      }    

   }

   cat("\n \n", "Independent categorical variables influencing the probability of the event : ", "\n \n")

   for(i in 1:length(Cure$Categorical)) {

   cat(labels_Cure$Categorical[[i]], " - reference value :", levels(Cure$Categorical[[i]])[1], "\n") 

   }  
 
   cat("\n")
   nalpha <- length(colnamesX) 

   # Definition of the design matrix X (cf. eq. 2.2) 
   W <- NULL 
   colnamesW <- c() 

   for(i in 1:length(Cox$Categorical)) {

      if( class(Cox$Categorical[[i]]) == "factor" ) {

         W <- cbind(W, model.matrix(~Cox$Categorical[[i]])[,-1])
         colnamesW <- c( colnamesW, paste(names(Cox$Categorical)[i], levels(Cox$Categorical[[i]])[-1], sep = "-") )

      }

   }

   cat("\n \n", "Independent categorical variables influencing the timing of the event for susceptible subjects : ", "\n \n")
 
   for(i in 1:length(Cox$Categorical)) {

   cat(labels_Cox$Categorical[[i]], " - reference value :", levels(Cox$Categorical[[i]])[1], "\n") 

   }   
   cat("\n")
   nbeta <- length(colnamesW)

   n_splines_Cov <- nknots_Cov + degree_Cov 

   BcovCure <- NULL 
   for(i in 1:length(Cure$Continuous)) { 

      # relocated and rescaled to be in [-1,1]
      tools_NUM <- ( max(Cure$Continuous[[i]]) + min(Cure$Continuous[[i]]) ) / 2 
      tools_DENOM <- ( max(Cure$Continuous[[i]]) - min(Cure$Continuous[[i]]) ) / 2 
      Cov_cont_OK <- ( Cure$Continuous[[i]] - tools_NUM ) / tools_DENOM

      tools_basis <- matrix(.Fortran("cubicBsplines", 
                            x = as.double(Cov_cont_OK), 
                            nx = as.integer(n), 
                            xl = as.double(-1), 
                            xr = as.double(1), 
                            ndx = as.integer(nknots_Cov), 
                            B = as.double(matrix(0, nrow = n, ncol = n_splines_Cov)))$B, ncol = n_splines_Cov)

      BcovCure <- cbind( BcovCure, tools_basis) 

   }

   cat("\n \n", "Independent continuous variables influencing the probability of the event: ", "\n \n")
 
   for(i in 1:length(Cure$Continuous)) {

   cat(labels_Cure$Continuous[[i]], "\n") 

   }  

   cat("\n")
   n_cov_cont_Cure <- length(Cure$Continuous)


   BcovCox <- NULL 
   for(i in 1:length(Cox$Continuous)) { 

      # relocated and rescaled to be in [-1,1]
      tools_NUM <- ( max(Cox$Continuous[[i]]) + min(Cox$Continuous[[i]]) ) / 2 
      tools_DENOM <- ( max(Cox$Continuous[[i]]) - min(Cox$Continuous[[i]]) ) / 2 
      Cov_cont_OK <- ( Cox$Continuous[[i]] - tools_NUM ) / tools_DENOM

      tools_basis <- matrix(.Fortran("cubicBsplines", 
                            x = as.double(Cov_cont_OK), 
                            nx = as.integer(n), 
                            xl = as.double(-1), 
                            xr = as.double(1), 
                            ndx = as.integer(nknots_Cov), 
                            B = as.double(matrix(0, nrow = n, ncol = n_splines_Cov)))$B, ncol = n_splines_Cov)

      BcovCox <- cbind( BcovCox, tools_basis) 

   }

   cat("\n \n", "Independent continuous variables influencing the timing of the event for susceptible subjects: ", "\n \n")
 
   for(i in 1:length(Cox$Continuous)) {

   cat(labels_Cox$Continuous[[i]], "\n") 

   }  

   cat("\n")
   n_cov_cont_Cox <- length(Cox$Continuous)
 

   # Roughness penalty matrix. 
   penalty_Cov <- matrix(.Fortran("penmat", 
                         nknots = as.integer(nknots_Cov), 
                         degree = as.integer(degree_Cov),
                         order = as.integer(order_Cov), 
                         res = as.double(matrix(0, ncol = n_splines_Cov, nrow = n_splines_Cov)))$res, ncol = n_splines_Cov)

   penalty_Cov <- penalty_Cov + diag(rep(1, n_splines_Cov))*10^(-6)


   index_phi_BD <- 1:n_splines_estimate_BD  
   index_phi_Cure <- ( index_phi_BD[length(index_phi_BD)] + 1 ) : ( index_phi_BD[length(index_phi_BD)] + n_cov_cont_Cure * n_splines_Cov )
   index_phi_Cox <- ( index_phi_Cure[length(index_phi_Cure)] + 1 ) : ( index_phi_Cure[length(index_phi_Cure)] + n_cov_cont_Cox * n_splines_Cov )
   index_alpha <- ( index_phi_Cox[length(index_phi_Cox)] + 1 ) : ( index_phi_Cox[length(index_phi_Cox)]+nalpha )
   index_beta <- ( index_alpha[length(index_alpha)] + 1 ) :  ( index_alpha[length(index_alpha)] + nbeta )
   index_pen_BD <- ( index_beta[length(index_beta)] + 1 ) :  ( index_beta[length(index_beta)] + 1 )
   index_pen_Cure <- ( index_pen_BD[length(index_pen_BD)] + 1 ) :  ( index_pen_BD[length(index_pen_BD)] + n_cov_cont_Cure )
   index_pen_Cox <- ( index_pen_Cure[length(index_pen_Cure)] + 1 ) :  (index_pen_Cure[length(index_pen_Cure)] + n_cov_cont_Cox )

   Post.chains <- Estimation( par = param,

	  	              index_phi_BD = index_phi_BD, 
               		      index_pen_BD = index_pen_BD, 
                              n_splines_BD = n_splines_BD, 
                              n_splines_estimate_BD = n_splines_estimate_BD, 
 	  	              Bobs = Bobs, 
     		              Bmiddle = Bmiddle,
     		              penalty_BD = penalty_BD,
 		              rank_penalty = rank_penalty, 

 		              bins = bins, 
 		              width = width, 
 		              upto = upto, 

 		              index_phi_Cure = index_phi_Cure, 
 		              index_pen_Cure = index_pen_Cure, 
	
 		              index_phi_Cox = index_phi_Cox, 
 		              index_pen_Cox = index_pen_Cox, 

        		      n_splines_Cov = n_splines_Cov, 
 		              n_cov_cont_Cure = n_cov_cont_Cure, 
 		              n_cov_cont_Cox = n_cov_cont_Cox, 
   			      BcovCure = BcovCure, 
   			      BcovCox = BcovCox, 
                              penalty_Cov = penalty_Cov,

                              index_alpha = index_alpha,
                              X = X, 

             	              index_beta = index_beta,
                              W = W, 
                     
                              Event = Event, 
                              n = n, 
  
                              nalpha = nalpha, 
                              nbeta = nbeta, 

                              iteration = iteration, 
                              burnin = burnin
                            )

   newDirectory <- paste(getwd(), paste(study.name, "Convergence checks", sep = "-"), sep = "/") 
   dir.create(newDirectory)

   Convergence.issues( alpha = Post.chains[["alphaPost"]], 
                       beta = Post.chains[["betaPost"]],

                       phi_BD = Post.chains[["phiPost_BD"]], 
                       pen_BD = Post.chains[["penPost_BD"]],
       	               delta_BD = Post.chains[["deltaPost_BD"]],

                       n_splines_estimate_BD = n_splines_estimate_BD,
 
                       phi_Cure = Post.chains[["phiPost_Cure"]], 
                       pen_Cure = Post.chains[["penPost_Cure"]],
       	               delta_Cure = Post.chains[["deltaPost_Cure"]],

                       phi_Cox = Post.chains[["phiPost_Cox"]], 
                       pen_Cox = Post.chains[["penPost_Cox"]],
       	               delta_Cox = Post.chains[["deltaPost_Cox"]],

		       n_cov_cont_Cure = n_cov_cont_Cure, 
		       n_cov_cont_Cox = n_cov_cont_Cox,
 		       n_splines_Cov = n_splines_Cov,

                       nalpha = nalpha, 
                       nbeta = nbeta,

                       accept = Post.chains[["accept"]], 
                       newDirectory = newDirectory 
                    )


   Nresults <- Results( study.name = study.name, 
		        alpha = Post.chains[["alphaPost"]],
                        colnamesX = colnamesX,
                        beta = Post.chains[["betaPost"]], 
                        colnamesW = colnamesW, 
                        phiCure = Post.chains[["phiPost_Cure"]],
                        continuous_cov_Cure = Cure$Continuous,
                        labels_Cure = labels_Cure,
                        phiCox = Post.chains[["phiPost_Cox"]],
                        continuous_cov_Cox = Cox$Continuous,
                        labels_Cox = labels_Cox,
                        n_cov_cont_Cure = n_cov_cont_Cure,
                        n_cov_cont_Cox = n_cov_cont_Cox,
                        nknots_Cov = nknots_Cov, 
                        n_splines_Cov = n_splines_Cov
                      )

   post.chains.OK <- list(  alphaPost = Post.chains[["alphaPost"]], 
                            betaPost  = Post.chains[["betaPost"]],
                            phiPost_BD = Post.chains[["phiPost_BD"]],
                            tau_BD = Post.chains[["penPost_BD"]],
                            delta_BD = Post.chains[["deltaPost_BD"]])

    for(i in 1:n_cov_cont_Cure) {
 
       post.chains.OK[[length(post.chains.OK)+1]] <- Post.chains[["phiPost_Cure"]][((i-1)*n_splines_Cov + 1):(i*n_splines_Cov), ]
       names(post.chains.OK)[length(post.chains.OK)] <- paste( names(Cure$Continuous)[i], "phi_Proba", sep = "-")

       post.chains.OK[[length(post.chains.OK)+1]] <- Post.chains[["penPost_Cure"]][i, ]
       names(post.chains.OK)[length(post.chains.OK)] <- paste( names(Cure$Continuous)[i], "tau_Proba", sep = "-")

       post.chains.OK[[length(post.chains.OK)+1]] <- Post.chains[["deltaPost_Cure"]][i, ]
       names(post.chains.OK)[length(post.chains.OK)] <- paste( names(Cure$Continuous)[i], "delta_Proba", sep = "-")

    }

    for(i in 1:n_cov_cont_Cox) {
 
       post.chains.OK[[length(post.chains.OK)+1]] <- Post.chains[["phiPost_Cox"]][((i-1)*n_splines_Cov + 1):(i*n_splines_Cov), ]
       names(post.chains.OK)[length(post.chains.OK)] <- paste( names(Cure$Continuous)[i], "phi_Timing", sep = "-")

       post.chains.OK[[length(post.chains.OK)+1]] <- Post.chains[["penPost_Cox"]][i, ]
       names(post.chains.OK)[length(post.chains.OK)] <- paste( names(Cure$Continuous)[i], "tau_Timing", sep = "-")

       post.chains.OK[[length(post.chains.OK)+1]] <- Post.chains[["deltaPost_Cox"]][i, ]
       names(post.chains.OK)[length(post.chains.OK)] <- paste( names(Cure$Continuous)[i], "delta_Timing", sep = "-")

    }

    res <- list(Nresults = Nresults, PostChains = post.chains.OK)
    return(res)
    
}

}
