##################################################################################################################
#
# Nonparametric double additive cure survival models: an application to the estimation of the nonlinear 
# effect of age at first parenthood on fertility progression.
# 
# Statistical Modelling - 2018
#
# Code writen by V.Bremhorst (Universite catholique de Louvain - Belgium)
#
# Joint work with : * Michaela Kreyenfeld - Hertie School of Governance Berlin (Germany) 					 			 
#                   * Philippe Lambert - Universite de Liege and Universite catholique de Louvain (Belgium) 
#
# Corresponding author: vincent.bremhorst@uclouvain.be 
#
#
#	     ********************************************************************************
#	     * Compute and return the numerical results for all regression parameters       *
#	     * Posterior median as estimator and the 90-95-99% HPD intervals 		    * 
#	     * Create the graphs illustrating the non-linear estimates of the effects of    *
#            * the continuous covariates                                                    *                                 
#	     ********************************************************************************
#
#
####################################################################################################################


Results <- function( study.name, 
		     alpha, 
                     colnamesX, 
                     beta, 
                     colnamesW, 
                     phiCure,
                     continuous_cov_Cure,
                     labels_Cure,
                     phiCox,
                     continuous_cov_Cox,
                     labels_Cox,
                     n_cov_cont_Cure, 
                     n_cov_cont_Cox, 
                     nknots_Cov, 
                     n_splines_Cov
                   )
{


   nalpha <- dim(alpha)[1]
   nbeta <- dim(beta)[1]

   alphaHPD1 <- HPDinterval(mcmc(t(alpha)), prob = 0.90)
   alphaHPD2 <- HPDinterval(mcmc(t(alpha)), prob = 0.95)
   alphaHPD3 <- HPDinterval(mcmc(t(alpha)), prob = 0.99)

   betaHPD1 <- HPDinterval(mcmc(t(beta)), prob = 0.90)
   betaHPD2 <- HPDinterval(mcmc(t(beta)), prob = 0.95)
   betaHPD3 <- HPDinterval(mcmc(t(beta)), prob = 0.99)

   Cnames <- c("Estimation", "post sd", "Lower90", "Upper90", "Lower95",  "Upper95", "Lower99", "Upper99")
   result <- matrix(nrow = nalpha+nbeta, ncol = length(Cnames)) 
   dimnames(result) <- list(c(colnamesX, colnamesW), Cnames)

   result[,1] <- c(apply(alpha, 1, median), apply(beta, 1, median)) 
   result[,2] <- c(apply(alpha, 1, sd), apply(beta,1,sd)) 
   result[,3] <- c(alphaHPD1[,1], betaHPD1[,1])
   result[,4] <- c(alphaHPD1[,2], betaHPD1[,2])
   result[,5] <- c(alphaHPD2[,1], betaHPD2[,1])
   result[,6] <- c(alphaHPD2[,2], betaHPD2[,2])
   result[,7] <- c(alphaHPD3[,1], betaHPD3[,1])
   result[,8] <- c(alphaHPD3[,2], betaHPD3[,2])

   filename <- paste("NumericalResults", "txt", sep = ".") 

   cat("\n \n")
   print(round(result,3))

   XXcov <- seq(-1, 1, by = 0.01)
   BcovXX <- matrix(.Fortran("cubicBsplines", 
                     x = as.double(XXcov), 
                     nx = as.integer(length(XXcov)), 
                     xl = as.double(-1), 
                     xr = as.double(1),
                     ndx = as.integer(nknots_Cov), 
                     B = as.double(matrix(0, nrow = length(XXcov), ncol = n_splines_Cov)))$B, ncol = n_splines_Cov)

   additiveCure <- list()
   additiveCureCI <- list()
   additiveCureAbscisse <- list() 
 
   for(i in 1:n_cov_cont_Cure) {

   additiveCure[[i]] <- BcovXX %*% phiCure[( (i-1)*n_splines_Cov + 1):(i*n_splines_Cov),]
   additiveCureCI[[i]] <- credible.region(t(additiveCure[[i]]), probs = 0.95)  

   tools_NUM <- ( max(continuous_cov_Cure[[i]]) + min(continuous_cov_Cure[[i]]) ) / 2 
   tools_DENOM <- ( max(continuous_cov_Cure[[i]]) - min(continuous_cov_Cure[[i]]) ) / 2 
   additiveCureAbscisse[[i]] <- XXcov * tools_DENOM +  tools_NUM

   Q025 <- quantile(continuous_cov_Cure[[i]], p = 0.025) 
   Q975 <- quantile(continuous_cov_Cure[[i]], p = 0.975) 
   keep <- (Q025 <= additiveCureAbscisse[[i]]) & (Q975 >= additiveCureAbscisse[[i]]) 

   dev.new()
   par(mfrow = c(1,1), oma = c(0,0,2,0))
   plot(additiveCureAbscisse[[i]][keep], apply(additiveCure[[i]], 1, median)[keep], type = "n", ylab = "",ylim = c(-2.5,2.5), xlab = labels_Cure$Continuous[[i]])
   title(paste(study.name, "Additive conditional effect on the probability", sep = "-"), line =3)
   title(labels_Cure$Continuous[[i]], line=1) 
   polygon(c(additiveCureAbscisse[[i]][keep], rev(additiveCureAbscisse[[i]][keep])), c(additiveCureCI[[i]][[1]][1,keep], rev(additiveCureCI[[i]][[1]][2,keep])), border = F, col = "grey")
   lines(additiveCureAbscisse[[i]][keep], apply(additiveCure[[i]], 1, median)[keep], col = 1, lty = 1, lwd = 2)
   abline(h = 0, col = 1, lty = 2, lwd = 2)

   }   

#*************************************************************

   additiveCox <- list()
   additiveCoxCI <- list()
   additiveCoxAbscisse <- list() 
 
   for(i in 1:n_cov_cont_Cox) {

   additiveCox[[i]] <- BcovXX %*% phiCox[( (i-1)*n_splines_Cov + 1):(i*n_splines_Cov),]
   additiveCoxCI[[i]] <- credible.region(t(additiveCox[[i]]), probs = 0.95)  

   tools_NUM <- ( max(continuous_cov_Cox[[i]]) + min(continuous_cov_Cox[[i]]) ) / 2 
   tools_DENOM <- ( max(continuous_cov_Cox[[i]]) - min(continuous_cov_Cox[[i]]) ) / 2 
   additiveCoxAbscisse[[i]] <- XXcov * tools_DENOM +  tools_NUM

   Q025 <- quantile(continuous_cov_Cox[[i]], p = 0.025) 
   Q975 <- quantile(continuous_cov_Cox[[i]], p = 0.975) 
   keep <- (Q025 <= additiveCoxAbscisse[[i]]) & (Q975 >= additiveCoxAbscisse[[i]]) 

   dev.new() 
   par(mfrow = c(1,1), oma = c(0,0,2,0))
   plot(additiveCoxAbscisse[[i]][keep], apply(additiveCox[[i]], 1, median)[keep], type = "n", ylab = "",ylim = c(-2.5,2.5), xlab = labels_Cox$Continuous[[i]])
   title(paste(study.name, "Additive conditional effect on the timing", sep = "-"), line =3)
   title(labels_Cox$Continuous[[i]], line=1) 
   polygon(c(additiveCoxAbscisse[[i]][keep], rev(additiveCoxAbscisse[[i]][keep])), c(additiveCoxCI[[i]][[1]][1,keep], rev(additiveCoxCI[[i]][[1]][2,keep])), border = F, col = "grey")
   lines(additiveCoxAbscisse[[i]][keep], apply(additiveCox[[i]], 1, median)[keep], col = 1, lty = 1, lwd = 2)
   abline(h = 0, col = 1, lty = 2, lwd = 2)

   }   

   return(result)

}

