##################################################################################################################
#
# Nonparametric double additive cure survival models: an application to the estimation of the nonlinear 
# effect of age at first parenthood on fertility progression.
# 
# Statistical Modelling - 2018
#
# Code writen by V.Bremhorst (Universite catholique de Louvain - Belgium)
#
# Joint work with : * Michaela Kreyenfeld - Hertie School of Governance Berlin (Germany) 					 			 
#                   * Philippe Lambert - Universite de Liege and Universite catholique de Louvain (Belgium) 
#
# Corresponding author: vincent.bremhorst@uclouvain.be 
#
#
#
#	**********************************************************************
#	* Perform the Bayesian estimation of the model         		     *
#	* Return the posterior median (as estimate) and the 90-95-99% HPD    *
#       * intervals of each regression parameter.			     *
#	* Return the posterior chains (after burnin) of all model parameters * 
#	**********************************************************************
#
#
####################################################################################################################


Estimation <- function( param, 

                        index_phi_BD, 
                        index_pen_BD, 
                        n_splines_BD, 
                        n_splines_estimate_BD, 
  	                Bobs, 
                        Bmiddle,
                        penalty_BD,
                        rank_penalty, 

			bins,
                        width, 
                        upto, 

                        index_phi_Cure, 
                        index_pen_Cure, 

                        index_phi_Cox, 
                        index_pen_Cox, 

     	                n_splines_Cov, 
                        n_cov_cont_Cure, 
                        n_cov_cont_Cox, 
                        BcovCure, 
                        BcovCox, 
                        penalty_Cov,

                        index_alpha,
                        X,

	                index_beta,
                        W, 
                     
                        Event,
                        n, 

                        nalpha, 
                        nbeta, 

                        iteration, 
                        burnin
                      )

{

   param <- rep(0, index_pen_Cox[length(index_pen_Cox)])

   cat("Frequentist estimation (based on the Augmented Lagrangian method) starts", "\n \n")

   op <- auglag( par = param, fn = log.post.c,

                 index_phi_BD = index_phi_BD, 
                 index_pen_BD = index_pen_BD, 
                 n_splines_BD = n_splines_BD, 
                 n_splines_estimate_BD = n_splines_estimate_BD, 
  	         Bobs = Bobs, 
                 Bmiddle = Bmiddle,
                 penalty_BD = penalty_BD,
                 rank_penalty = rank_penalty, 

                 width = width, 
                 upto = upto, 

                 index_phi_Cure = index_phi_Cure, 
                 index_pen_Cure = index_pen_Cure, 

                 index_phi_Cox = index_phi_Cox, 
                 index_pen_Cox = index_pen_Cox, 

	         n_splines_Cov = n_splines_Cov, 
                 n_cov_cont_Cure = n_cov_cont_Cure, 
                 n_cov_cont_Cox = n_cov_cont_Cox, 
                 BcovCure = BcovCure, 
                 BcovCox = BcovCox, 
                 penalty_Cov = penalty_Cov,

                 index_alpha = index_alpha,
                 X = X, 

	         index_beta = index_beta,
                 W = W, 
                     
                 Event = Event, 
                 heq = heq,  
                 control.outer = list(trace = FALSE))

   if (op$convergence == 0) {

   cat("End of the frequentist step.", "\n\n")

   }

   else {

   cat("Convergence problem(s)", "\n\n")

   }

   phi_BD_Init <- c(op$par[index_phi_BD],rep(10,n_splines_BD - n_splines_estimate_BD))
   pen_BD_Init <- exp(op$par[index_pen_BD])

   phi_Cure_Init <- op$par[index_phi_Cure]
   pen_Cure_Init <- exp(op$par[index_pen_Cure])

   phi_Cox_Init <- op$par[index_phi_Cox]
   pen_Cox_Init <- exp(op$par[index_pen_Cox])

   alpha_Init <- op$par[index_alpha]
   beta_Init <- op$par[index_beta]

   sigma <- solve(op$hessian)
   sigma_Cure <- sigma[c(index_phi_Cure, index_alpha[1]), c(index_phi_Cure, index_alpha[1])]   
   sigma_Cox <- sigma[c(index_phi_BD, index_phi_Cox), c(index_phi_BD, index_phi_Cox)]

   cat("Start of the MCMC algorithm", "\n \n")

   res = .Fortran("FPTRC", # Flexible Promotion time for right censored data 

                   n_splines_BD = as.integer(n_splines_BD),    
                   n_splines_estimate_BD = as.integer(n_splines_estimate_BD),  
                   rank_penalty = as.integer(rank_penalty),               
 
                   n_splines_Cov_Cure = as.integer(n_splines_Cov*n_cov_cont_Cure),  
                   n_splines_Cov_Cox = as.integer(n_splines_Cov*n_cov_cont_Cox),  
                   n_splines_Cov_one = as.integer(n_splines_Cov),
 
                   n_cov_cont_Cure = as.integer(n_cov_cont_Cure),                
                   n_cov_cont_Cox = as.integer(n_cov_cont_Cox),                

                   Bobs = as.double(Bobs), 
                   Bmiddle = as.double(Bmiddle), 
                   BcovCure = as.double(BcovCure),  
                   BcovCox = as.double(BcovCox),  

                   penalty_BD = as.double(penalty_BD), 
                   penalty_Cov = as.double(penalty_Cov),

                   sigma_Cure = as.double(sigma_Cure),
                   sigma_Cox = as.double(sigma_Cox), 
                   
                   n = as.integer(n),
                   eventInd = as.integer(Event), 
 
                   Npart = as.integer(bins),
                   delta = as.double(width),
                   upto = as.integer(upto), 

                   X = as.double(X),
                   W = as.double(W),   
                   nalpha = as.integer(nalpha),
                   nbeta = as.integer(nbeta),          

                   nu = as.double(2), 
                   a_delta = as.double(0.0001),  
                   b_delta = as.double(0.0001), 
                   sd_Cov = as.double(10),

                   initphi_BD = as.double(phi_BD_Init),
                   initpen_BD = as.double(pen_BD_Init), 
                   initdelta_BD = as.double(1), 

                   initalpha = as.double(alpha_Init),
   
                   initphi_Cure = as.double(phi_Cure_Init), 
                   initpen_Cure = as.double(pen_Cure_Init), 
                   initdelta_Cure = as.double(1), 

                   initbeta = as.double(beta_Init),
                    
                   initphi_Cox = as.double(phi_Cox_Init), 
                   initpen_Cox = as.double(pen_Cox_Init), 
                   initdelta_Cox = as.double(1), 

                   InitSDprop_Cure = as.double(0.1),
                   InitSDprop_Cox = as.double(0.1),
                   InitSDprop_alpha = as.double(0.1), 
 
                   iteration = as.integer(iteration),
                   updatesd = as.integer(burnin), 

                   phi_BD = as.double(matrix(0, ncol =  iteration, nrow = n_splines_BD)), 
                   pen_BD = as.double(rep(0, iteration+1)),             
                   delta_BD = as.double(rep(0, iteration+1)),                    
 
                   alpha = as.double(matrix(0, ncol = iteration, nrow = nalpha)), 

                   phi_Cure = as.double(matrix(0, nrow = n_cov_cont_Cure*n_splines_Cov, ncol = iteration)),
                   pen_Cure = as.double(matrix(0, nrow = n_cov_cont_Cure, ncol = iteration+1)), 
                   delta_Cure = as.double(matrix(0, nrow = n_cov_cont_Cure, ncol = iteration+1)),
 
                   beta = as.double(matrix(0, ncol = iteration, nrow = nbeta)), 
             
                   phi_Cox = as.double(matrix(0, nrow = n_cov_cont_Cox*n_splines_Cov, ncol = iteration)),
                   pen_Cox = as.double(matrix(0, nrow = n_cov_cont_Cox, ncol =  iteration+1)), 
                   delta_Cox = as.double(matrix(0, nrow = n_cov_cont_Cox, ncol = iteration+1)),

                   accept = as.integer(matrix(0, ncol = iteration, nrow = 3)))

   cat("End of the MCMC algorithm", "\n \n")

   accept <- matrix(res$accept, ncol = iteration)[,-c(1:burnin)]

   phiPost_BD <- matrix(res$phi_BD, ncol = iteration)[,-c(1:burnin)]
   phiPost_theta <- matrix(res$phi_Cure, ncol = iteration)[,-c(1:burnin)]
   phiPost_Cox <- matrix(res$phi_Cox, ncol = iteration)[,-c(1:burnin)]

   alphaPost <- matrix(res$alpha, ncol = iteration)[,-c(1:burnin)]
   betaPost <- matrix(res$beta, ncol = iteration)[,-c(1:burnin)]

   penPost_BD <- res$pen_BD[-c(1:(burnin+1))]
   deltaPost_BD <- res$delta_BD[-c(1:(burnin+1))]

   penPost_theta <- matrix(res$pen_Cure, ncol = iteration+1)[,-c(1:(burnin+1))]
   deltaPost_theta <- matrix(res$delta_Cure, ncol = iteration+1)[,-c(1:(burnin+1))]

   if (n_cov_cont_Cure == 1) {

      penPost_theta <- t(penPost_theta)
      deltaPost_theta <- t(deltaPost_theta)

   }

   penPost_Cox <- matrix(res$pen_Cox, ncol = iteration+1)[,-c(1:(burnin+1))]
   deltaPost_Cox <- matrix(res$delta_Cox, ncol = iteration+1)[,-c(1:(burnin+1))]

   if (n_cov_cont_Cox == 1) {

      penPost_Cox <- t(penPost_Cox)
      deltaPost_Cox <- t(deltaPost_Cox)

   }

   phiPost10_BD <- matrix(0, ncol = (iteration-burnin)/10, nrow = n_splines_BD)
   phiPost10_theta <- matrix(0, ncol = (iteration-burnin)/10, nrow = n_cov_cont_Cure*n_splines_Cov)
   phiPost10_Cox <- matrix(0, ncol = (iteration-burnin)/10, nrow = n_cov_cont_Cox*n_splines_Cov)

   alphaPost10 <- matrix(0, ncol = (iteration-burnin)/10, nrow = nalpha)
   betaPost10 <- matrix(0, ncol = (iteration-burnin)/10, nrow = nbeta)

   penPost10_BD <- c()
   deltaPost10_BD <- c()

   penPost10_theta <- matrix(0, ncol = (iteration-burnin)/10, nrow = n_cov_cont_Cure) 
   deltaPost10_theta <- matrix(0, ncol = (iteration-burnin)/10, nrow = n_cov_cont_Cure)
      
   penPost10_Cox <- matrix(0, ncol = (iteration-burnin)/10, nrow = n_cov_cont_Cox)
   deltaPost10_Cox <- matrix(0, ncol = (iteration-burnin)/10, nrow = n_cov_cont_Cox)

   for(i in 1:((iteration-burnin)/10)) {

      phiPost10_BD[,i] = phiPost_BD[,((i-1)*10) +1]
      phiPost10_theta[,i] = phiPost_theta[,((i-1)*10) +1]
      phiPost10_Cox[,i] = phiPost_Cox[,((i-1)*10) +1]

      alphaPost10[,i] = alphaPost[,((i-1)*10) +1] 
      betaPost10[,i] = betaPost[,((i-1)*10) +1]

      penPost10_BD[i] = penPost_BD[((i-1)*10) +1]
      deltaPost10_BD[i] = deltaPost_BD[((i-1)*10) +1]

      penPost10_theta[,i] = penPost_theta[,((i-1)*10) +1]
      deltaPost10_theta[,i] = deltaPost_theta[,((i-1)*10) +1]
 
      penPost10_Cox[,i] = penPost_Cox[,((i-1)*10) +1]
      deltaPost10_Cox[,i] = deltaPost_Cox[,((i-1)*10) +1]

   }

   return ( list( phiPost_BD = phiPost10_BD, 
                 phiPost_Cure = phiPost10_theta, 
                 phiPost_Cox = phiPost10_Cox, 
    
                 alphaPost = alphaPost10, 
 	  	 betaPost = betaPost10,

	         penPost_BD = penPost10_BD,
                 deltaPost_BD = deltaPost10_BD,

                 penPost_Cure = penPost10_theta,
                 deltaPost_Cure = deltaPost10_theta,
 
                 penPost_Cox = penPost10_Cox,
                 deltaPost_Cox = deltaPost10_Cox,  
 
                 accept = accept) 
          )
} 

